/** @par LICENCE
* @verbatim
Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.

ADDI-DATA GmbH
Airpark Business Center
Airport Boulevard B210
77836 Rheinm�nster
Germany
Tel: +49(0)7229/1847-0
Fax: +49(0)7229/1847-200
http://www.addi-data-com
info@addi-data.com

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You shoud find the complete GPL in the COPYING file accompanying
this source code.
* @endverbatim
*/

/** @mainpage Information about the APCIE-1516 Linux driver.
*
* @section intro_sec Introduction
*
* This documentation is about the APCIE-1516 Linux driver.
* With this driver it is possible to:
* - Read port state.
* - Set port state.
*/

#include "apcie1516-private.h"
#include "vtable.h"

#include "knowndev.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/sysfs.h>
#endif

#include <apcie1516-kapi.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("ADDI-DATA GmbH <info@addi-data.com>");
MODULE_DESCRIPTION("APCIE-1516 IOCTL driver");

EXPORT_SYMBOL(APCIE1516_Read8DigitalInputs);
EXPORT_SYMBOL(APCIE1516_Set8DigitalOutputsOn);
EXPORT_SYMBOL(APCIE1516_Set8DigitalOutputsOff);
EXPORT_SYMBOL(APCIE1516_SetDigitalOutputMemoryOn);
EXPORT_SYMBOL(APCIE1516_SetDigitalOutputMemoryOff);
EXPORT_SYMBOL(APCIE1516_Get8DigitalOutputStatus);

EXPORT_NO_SYMBOLS;

#ifndef __devinit
#define __devinit
#define __devexit
#define __devinitdata
#define __devexit_p
#endif

//------------------------------------------------------------------------------

atomic_t apcie1516_count = ATOMIC_INIT(0);
unsigned int apcie1516_majornumber = 0;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static struct class * apcie1516_class = NULL;
#endif

//------------------------------------------------------------------------------
static int __devinit apcie1516_probe_one(struct pci_dev *dev,const struct pci_device_id *ent);
static void __devexit apcie1516_remove_one(struct pci_dev *dev);

//------------------------------------------------------------------------------
/** The ID table is an array of struct pci_device_id ending with a all-zero entry.
Each entry consists of:

- vendor, device	Vendor and device ID to match (or PCI_ANY_ID)
- subvendor,	Subsystem vendor and device ID to match (or PCI_ANY_ID)
- subdevice class,		Device class to match. The class_mask tells which bits
- class_mask	of the class are honored during the comparison.
- driver_data	Data private to the driver.
*/

static struct pci_device_id apcie1516_pci_tbl[] __devinitdata = {
		{
				APCIE1516_BOARD_VENDOR_ID,
				APCIE1516_BOARD_DEVICE_ID,
				PCI_ANY_ID,
				PCI_ANY_ID,
				0,
				0,
				0
		},
		{ 0 },	/* terminate list */
};

MODULE_DEVICE_TABLE (pci, apcie1516_pci_tbl);

//----------------------------------------------------------------------------------------------------------------------
static struct pci_driver apcie1516_pci_driver = {
		.name		= __DRIVER_NAME,
		.probe		= apcie1516_probe_one,
		.remove		= __devexit_p(apcie1516_remove_one),
		.id_table	= apcie1516_pci_tbl,
};
//------------------------------------------------------------------------------
static struct file_operations apcie1516_fops =
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
		.ioctl		= apcie1516_ioctl_lookup,
#else
		.unlocked_ioctl	= apcie1516_ioctl_lookup,
#endif
		.open		= apcie1516_open_lookup,
		.release	= apcie1516_release_lookup,
};

//------------------------------------------------------------------------------
/** when module is unloaded, stop all board activities (cf interrupt)*/
static void apcie1516_stop_board(struct pci_dev * pdev)
{
	// FILL THE BLANK
}

//-------------------------------------------------------------------
/** event: a new card is detected.
*
* Historically - in former ADDI-DATA drivers - data about a board has been stored
* in a static structure.
* This led to huge duplication of information since most of these information
* are already present in the pci_dev struct.
*
* Now we manipulate this standard structure provided by the OS and we attach
* private data using the field driver_data, if necessary.
*
*  */
static int __devinit apcie1516_probe_one(struct pci_dev *dev, const struct pci_device_id *ent)
{
	{
		int ret = pci_enable_device(dev);
		if (ret)
		{
			printk(KERN_ERR "%s: pci_enable_device failed\n",__DRIVER_NAME);
			return ret;
		}
	}

	/* 2.4 only : check_region is deprecated in 2.6 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	/* check if BAR 0 is free */
	if ( check_region(dev->resource[0].start, pci_resource_len(dev,0)) )
	{
		printk(KERN_WARNING "%s: BAR 0 (%lX) is already in use\n",__DRIVER_NAME,dev->resource[0].start);
		return -EBUSY;
	}
	/* check if BAR 1 is free */
	if ( check_region(dev->resource[1].start, pci_resource_len(dev,1)) )
	{
		printk(KERN_WARNING "%s: BAR 1 (%lX) is already in use\n",__DRIVER_NAME,dev->resource[1].start);
		return -EBUSY;
	}
	/* check if BAR 2 is free */
	if ( check_region(dev->resource[2].start, pci_resource_len(dev,2)) )
	{
		printk(KERN_WARNING "%s: BAR 2 (%lX) is already in use\n",__DRIVER_NAME,dev->resource[2].start);
		return -EBUSY;
	}
	/* check if BAR 3 is free */
	if ( check_region(dev->resource[3].start, pci_resource_len(dev,3)) )
	{
		printk(KERN_WARNING "%s: BAR 3 (%lX) is already in use\n",__DRIVER_NAME,dev->resource[3].start);
		return -EBUSY;
	}
#endif // 2.4

	/* allocate a new data structure containing board private data */
	{
		struct apcie1516_str_BoardInformations * newboard_data = NULL;
		newboard_data = kmalloc( sizeof( struct apcie1516_str_BoardInformations) , GFP_ATOMIC);
		if (!newboard_data)
		{
			printk(KERN_CRIT "Can't allocate memory for new board %s\n",pci_name(dev));
			return -ENOMEM;
		}
		/* link standard data structure to the board's private one */
		pci_set_drvdata(dev,newboard_data);

		apcie1516_init_priv_data(newboard_data);
	}

	/* lock BAR IO ports ressources */
	{
		int ret = pci_request_regions(dev,__DRIVER_NAME);
		if (ret)
		{
			printk(KERN_ERR "%s: pci_request_regions failed\n",__DRIVER_NAME);
			/* free all allocated ressources here*/
			kfree(APCIE1516_PRIVDATA(dev));
			return ret;
		}
	}

	{
		/* increase the global board count */
		atomic_inc(&apcie1516_count);
		printk(KERN_INFO "%s: board %s managed (minor number will be %d) \n",__DRIVER_NAME, pci_name(dev), atomic_read(&apcie1516_count)-1);
	}

	/* create /proc entry */
	apcie1516_proc_create_device(dev, atomic_read(&apcie1516_count)-1);

	apcie1516_known_dev_append(dev);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	while(1)
	{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,26)
		struct class_device *cdev;
#else
		struct device *cdev;
#endif

		int minor = (atomic_read(&apcie1516_count)-1);

        /* don't execute if class not exists */
        if (IS_ERR(apcie1516_class))
                break;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,26)
		cdev = class_device_create (apcie1516_class, NULL, MKDEV(apcie1516_majornumber, minor), NULL, "%s_%d", __DRIVER_NAME, minor);
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,27)
		cdev = device_create (apcie1516_class, NULL, MKDEV(apcie1516_majornumber, minor), "%s_%d", __DRIVER_NAME, minor);
#else
		cdev = device_create (apcie1516_class, NULL, MKDEV(apcie1516_majornumber, minor), NULL, "%s_%d", __DRIVER_NAME, minor);
#endif

		if (IS_ERR(cdev))
		{
			printk (KERN_WARNING "%s: class_device_create error\n", __DRIVER_NAME);
		}
		break;
	}
#endif

	return 0;
}


//------------------------------------------------------------------------------
/** This function registers the driver with register_chrdev.
*
* @todo For kernel 2.6, use udev
*  */

static int apcie1516_register(void)
{

	//Registration of driver
	apcie1516_majornumber = register_chrdev(0, __DRIVER_NAME, &apcie1516_fops);

	if (apcie1516_majornumber < 0)
	{
		printk (KERN_ERR "%s: register_chrdev returned %d ... aborting\n",__DRIVER_NAME,apcie1516_majornumber);
		return -ENODEV;
	}
	return 0;
}

//------------------------------------------------------------------------------
/** Called when module loads. */
static int __init apcie1516_init(void)
{

	/* 2.4 only : check_region is deprecated in 2.6 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	/* Detection if an PCI Bus is present */
	if(!pci_present())
	{
		printk(KERN_WARNING "%s: NO PCI BUS.\n", __DRIVER_NAME);
		return -ENODEV;
	}
#endif // 2.4

	apcie1516_init_vtable(apcie1516_vtable);

	if( apcie1516_register() )
	{
		return -ENODEV;
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	apcie1516_class = ADDI_CLASS_CREATE(THIS_MODULE, __DRIVER_NAME);
	if (IS_ERR(apcie1516_class))
	{
		printk (KERN_WARNING "%s: class_create error\n",__DRIVER_NAME );
	}
#endif


	/* registred, now create root /proc entry */
	apcie1516_proc_init();

	printk(KERN_INFO "%s: loaded\n",__DRIVER_NAME);

	/* now, subscribe to PCI bus subsystem  */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18)
	/** @note The module will load anyway even if the call to pci_module_init() fails */
	if ( pci_module_init (&apcie1516_pci_driver) != 0)
#else
		if ( pci_register_driver (&apcie1516_pci_driver) != 0)
#endif
		{
			printk("%s: no device found\n",__DRIVER_NAME);
			apcie1516_pci_driver.name = "";
		}

	return 0;
}

//-------------------------------------------------------------------
/** event: a card is removed (also called when module is unloaded) */
static void __devexit apcie1516_remove_one(struct pci_dev *dev)
{

	/* stop board activities */
	apcie1516_stop_board(dev);

	/* do OS-dependant thing we don't really want to know of :) */
	pci_disable_device(dev);

	/* deallocate BAR IO Ports ressources */
	pci_release_regions(dev);

	apcie1516_known_dev_remove(dev);

	/* free private device data */
	kfree(APCIE1516_PRIVDATA(dev));

	/* delete associated /proc entry */
	apcie1516_proc_release_device(dev);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	while(1)
	{
		int minor = (atomic_read(&apcie1516_count)-1);

		/* don't execute if class not exists */
		if(IS_ERR(apcie1516_class))
				break;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,26)
		class_device_destroy (apcie1516_class, MKDEV(apcie1516_majornumber, minor));
#else
		device_destroy (apcie1516_class, MKDEV(apcie1516_majornumber, minor));
#endif

		break;
	}
#endif

	/* decrement global board count*/
	atomic_dec(&apcie1516_count);

	printk("%s: board %s unloaded \n",__DRIVER_NAME,pci_name(dev));

}

//------------------------------------------------------------------------------
/** This function unregisters the driver with unregister_chrdev.
*
* @todo For kernel 2.6, use udev
*  */

static void apcie1516_unregister(void)
{
	unregister_chrdev(apcie1516_majornumber, __DRIVER_NAME);
}

//-------------------------------------------------------------------
/** Called when module is unloaded. */
static void __exit apcie1516_exit(void)
{

	/* unsubscribe to PCI bus subsystem */
	if (apcie1516_pci_driver.name[0])
	{
		pci_unregister_driver (&apcie1516_pci_driver);
	}

	/* unsubscribe to /dev VFS */
	apcie1516_unregister();

	/* delete /proc root */
	apcie1516_proc_release();

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	class_destroy (apcie1516_class);
#endif

}
//------------------------------------------------------------------------------
module_exit(apcie1516_exit);
module_init(apcie1516_init);
