 /* @file apcie1516-kapi.h
  * 
  * @brief Define the kernel API for the apcie1516 ADDI-DATA board. 
  * 
  * @author Sylvain Nahas, Julien Krauth
  * 
  */

/** @par LICENCE
 * 
 *  @verbatim
   Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
   ADDI-DATA GmbH
   Airpark Business Center
   Airport Boulevard B210
   77836 Rheinmnster
   Germany
   Tel: +49(0)7229/1847-0
   Fax: +49(0)7229/1847-200
   http://www.addi-data-com
   info@addi-data.com
 
 This library is free software; you can redistribute it and/or modify it under 
 the terms of the GNU Lesser General Public License as published by the 
 Free Software Foundation; either version 2.1 of the License, 
 or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful, 
 but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 or FITNESS FOR A PARTICULAR PURPOSE. 
 See the GNU Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public License 
 along with this library; if not, write to the Free Software Foundation, 
 Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 
 
 You also shoud find the complete LGPL in the LGPL.txt file accompanying 
 this source code.
 
 @endverbatim
**/


#ifndef __APCIE1516KAPI_H__
#define __APCIE1516KAPI_H__

#ifndef __KERNEL__
#error This header is intended to be used in kernel context only.
#endif

#ifdef __KERNEL__

#include <linux/pci.h>
#include <apcie1516.h>

/** Read all digital inputs of the APCI-1516.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [out]		arg		Function argument. An unsigned char that will contain port state.
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
int APCIE1516_Read8DigitalInputs(struct pci_dev *pdev, uint8_t * PortState);

/** Set all digital outputs of the APCI-1516 on.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		arg		Function argument. An unsigned char that has contain port state to set.
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
int APCIE1516_Set8DigitalOutputsOn(struct pci_dev *pdev, uint8_t arg);


/** Set all digital outputs of the APCI-1516 off.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE1516_Set8DigitalOutputsOn when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param [in]		arg		Function argument. An unsigned char that has contain port state to reset.
 *
 * @retval 0		No error. 
 * @retval -EPERM	Output memory is not activated.
 * @retval -EFAULT	Can't return parameters.
 */
int APCIE1516_Set8DigitalOutputsOff(struct pci_dev *pdev, unsigned long arg);

/** Set the digital output memory on.
 *
 * @param [in,out]	pdev	Pointer on the device structure. *
 * @retval 0		No error.
 */
int APCIE1516_SetDigitalOutputMemoryOn(struct pci_dev *pdev);

/** Set the digital output memory off.
 *
 * @param [in,out]	pdev	Pointer on the device structure.

 * @retval 0		No error.
 */
int APCIE1516_SetDigitalOutputMemoryOff(struct pci_dev *pdev);

/** Get all digital outputs states of the APCI-1516.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [out]		arg		Function argument. An unsigned char that contain port state.
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
int APCIE1516_Get8DigitalOutputStatus(struct pci_dev *pdev, uint8_t * PortState);

/** find a apci 1516 device of a given index in the system PCI device list.
 * @param index The index (minor number) to lookup
 * @return A pointer to the device or NULL
 *
 * This function is used to map a minor number to an actual device.
 */
struct pci_dev * apcie1516_lookup_board_by_index(unsigned int index);

#endif// __KERNEL__


#endif /*__APCIE1516KAPI_H__*/
