/** @file sample_apcie2200_digital_io.c
 
   This sample demonstartes how to use digital input / output in user mode.
 
   @par CREATION  
   @author Krauth Julien
   @date   28.01.2010
   
   @par VERSION
   @verbatim
   $LastChangedRevision:$
   $LastChangedDate:$
   @endverbatim   
   
   @par LICENCE
   @verbatim
   Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
   ADDI-DATA GmbH
   Airpark Business Center
   Airport Boulevard B210
   77836 Rheinmuenster
   Germany
   Tel: +49(0)7229/1847-0
   Fax: +49(0)7229/1847-200
   http://www.addi-data-com
   info@addi-data.com
        
   This program is free software; you can redistribute it and/or modify it under 
   the terms of the GNU General Public License as published by the Free Software 
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. 
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with 
   this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   You shoud also find the complete GPL in the COPYING file 
   accompanying this source code.
   @endverbatim   
 */


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <sys/sysmacros.h> /* major() minor() */

#include <signal.h> /* For signal */

#include <apcie2200.h>
#include <apcie2200-events.h>

#include "findboards.h"

/* This global variable is used to stop each single samples */
unsigned char ucStopFlag;

//------------------------------------------------------------------------------

/** Move the consol cursor on the specified point.
*
* @param x	Horizontal position of the cursor.
* @param y	Vertical position of the cursor.
*
* retval 0 No error.
*/
int gotoxy(int x, int y)
{
	char essq[100]; // String variable to hold the escape sequence
	char xstr[100]; // Strings to hold the x and y coordinates
	char ystr[100]; // Escape sequences must be built with characters

	// Convert the screen coordinates to strings
	sprintf(xstr, "%d", x);
	sprintf(ystr, "%d", y);

	// Build the escape sequence (vertical move)
	essq[0] = '\0';
	strcat(essq, "\033[");
	strcat(essq, ystr);

	// Described in man terminfo as vpa=\\E[%p1%dd
	// Vertical position absolute
	strcat(essq, "d");

	// Horizontal move
	// Horizontal position absolute
	strcat(essq, "\033[");
	strcat(essq, xstr);

	// Described in man terminfo as hpa=\\E[%p1%dG
	strcat(essq, "G");

	// Execute the escape sequence
	// This will move the cursor to x, y
	printf("%s", essq);

	return 0;
}

//------------------------------------------------------------------------------

/** Test APCIE-2200 digital input.
*
* @param [in]	fd	File descriptor of the board to use.
*
* @retval 0	No error.
* @retval 1	Error.
*/
int apcie2200_test_digital_input (int fd)
{
	uint32_t PortState = 0;
	uint32_t nbrOfInputs = 0;

	/* Clean the screen before display input status */
	system("clear");

	printf ("CMD_APCIE2200_GetNumberOfDigitalInputs\n");
	if (ioctl (fd, CMD_APCIE2200_GetNumberOfDigitalInputs, &nbrOfInputs) < 0)
	{
		perror("ioctl");
	}

	while (ucStopFlag == 0)
	{
		/* Read all digital input state */
		printf ("CMD_APCIE2200_Read16DigitalInputs\n");
		if ( ioctl (fd,CMD_APCIE2200_Read16DigitalInputs,&PortState) < 0 )
		{
			perror("ioctl");
			exit(1);
		}

		/* Set the cursor on screen top */
		gotoxy (0,0);
		printf ("\n+--------------------------------------------------------------------------------------+");
		printf ("\n| DIGITAL INPUTS STATES  (Number of digital inputs: %02i)                                |", nbrOfInputs);
		printf ("\n+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+------+");
		printf ("\n| 16 | 15 | 14 | 13 | 12 | 11 | 10 | 9  | 8  | 7  | 6  | 5  | 4  | 3  | 2  | 1  | Hex  |");
		printf ("\n+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+------+");
		printf ("\n|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s| %4x |",
				(PortState & 32768)?"ON":"OFF",
				(PortState & 16384)?"ON":"OFF",
				(PortState & 8192)?"ON":"OFF",
				(PortState & 4096)?"ON":"OFF",
				(PortState & 2048)?"ON":"OFF",
				(PortState & 1024)?"ON":"OFF",
				(PortState & 512)?"ON":"OFF",
				(PortState & 256)?"ON":"OFF",		
				(PortState & 128)?"ON":"OFF",
				(PortState & 64)?"ON":"OFF",
				(PortState & 32)?"ON":"OFF",
				(PortState & 16)?"ON":"OFF",
				(PortState & 8)?"ON":"OFF",
				(PortState & 4)?"ON":"OFF",
				(PortState & 2)?"ON":"OFF",
				(PortState & 1)?"ON":"OFF",
				PortState);
		printf ("\n+--------------------------------------------------------------------------------------+");
		printf ("\n");
		printf ("\n");
		printf ("ctrl + c for the next test\n");
		printf ("\n");
		usleep (250000);
	}

	return 0;
}

//------------------------------------------------------------------------------

/** Test APCIE-2200 digital output.
*
* @param [in]	fd	File descriptor of the board to use.
*
* @retval 0	No error.
* @retval 1	Error.
*/
int apcie2200_test_digital_output (int fd)
{
	uint32_t PortState = 0;
	uint32_t nbrOfOutputs = 0;
	int i = 0;

	printf ("CMD_APCIE2200_GetNumberOfDigitalOutputs\n");
	if (ioctl (fd, CMD_APCIE2200_GetNumberOfDigitalOutputs, &nbrOfOutputs) < 0)
	{
		perror("ioctl");
	}

	/* Set the digital output memory on.<br>
	 * When digital output memory is activated,<br>
	 * the previous output state is hold when one<br>
	 * or more output states are set.
	 */
	printf ("CMD_APCIE2200_SetDigitalOutputMemoryOn\n");
	if ( ioctl (fd,CMD_APCIE2200_SetDigitalOutputMemoryOn,NULL) < 0 )
	{
		perror("ioctl");
		exit(1);
	}

	printf ("\n");
	printf ("\n");

	while (ucStopFlag == 0)
	{
		usleep (250000);

		if (i < nbrOfOutputs)
		{
			PortState = 1 << i;

			printf ("CMD_APCIE2200_Set16DigitalOutputsOn\n");
			if ( ioctl (fd,CMD_APCIE2200_Set16DigitalOutputsOn,PortState) < 0 )
			{
				perror("ioctl");
				exit(1);
			}
		}

		if (i > (nbrOfOutputs-1))
		{
			PortState = 1 << (i-nbrOfOutputs);

			printf ("CMD_APCIE2200_Set16DigitalOutputsOff\n");
			if ( ioctl (fd,CMD_APCIE2200_Set16DigitalOutputsOff,PortState) < 0 )
			{
				perror("ioctl");
				exit(1);
			}
		}

		printf ("CMD_APCIE2200_Get16DigitalOutputStatus\n");
		if ( ioctl (fd,CMD_APCIE2200_Get16DigitalOutputStatus,&PortState) < 0 )
		{
			perror("ioctl");
			exit(1);
		}

		gotoxy (0,0);
		printf ("\n+--------------------------------------------------------------------------------------+");
		printf ("\n| DIGITAL OUTPUTS STATES  (Number of digital inputs: %02i)                               |",nbrOfOutputs);
		printf ("\n+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+------+");
		printf ("\n| 16 | 15 | 14 | 13 | 12 | 11 | 10 | 9  | 8  | 7  | 6  | 5  | 4  | 3  | 2  | 1  | Hex  |");
		printf ("\n+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+----+------+");
		printf ("\n|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s|%4s| %4x |",
				(PortState & 32768)?"ON":"OFF",
				(PortState & 16384)?"ON":"OFF",
				(PortState & 8192)?"ON":"OFF",
				(PortState & 4096)?"ON":"OFF",
				(PortState & 2048)?"ON":"OFF",
				(PortState & 1024)?"ON":"OFF",
				(PortState & 512)?"ON":"OFF",
				(PortState & 256)?"ON":"OFF",		
				(PortState & 128)?"ON":"OFF",
				(PortState & 64)?"ON":"OFF",
				(PortState & 32)?"ON":"OFF",
				(PortState & 16)?"ON":"OFF",
				(PortState & 8)?"ON":"OFF",
				(PortState & 4)?"ON":"OFF",
				(PortState & 2)?"ON":"OFF",
				(PortState & 1)?"ON":"OFF",
				PortState);
		printf ("\n+--------------------------------------------------------------------------------------+");
		printf ("\n");
		printf ("\n");
		
		if ( i == ((nbrOfOutputs * 2) - 1) )
			i = 0;
		else
			i++;		
	}

	printf ("\n");

	/* Set the digital output memory off */
	printf ("CMD_APCIE2200_SetDigitalOutputMemoryOff\n");
	if ( ioctl (fd,CMD_APCIE2200_SetDigitalOutputMemoryOff,NULL) < 0 )
	{
		perror("ioctl");
		exit(1);
	}

	return 0;
}

//------------------------------------------------------------------------------

void v_ControlExit (int i)
{
	ucStopFlag=1;
}

//------------------------------------------------------------------------------

int main(int argc, char** argv)
{
	int           apcie2200_card_fd;
	unsigned char ucTestInputs     ;
	
	if (argc < 3 )
	{
                printf("Not enough arguments.\n");
		printf("Usage: %s /dev/apcie2200_[0..15] input|output\n", argv[0]);
		exit(1);
	}

	if ( strcmp(argv[2], "input") == 0 )
	{
		ucTestInputs = 1;
	}
	else if ( strcmp(argv[2], "output") == 0 )
	{
		ucTestInputs = 0;
	}
	else
	{
		printf("Wrong parameter. Must be input or output.\n");
		printf("Usage: %s /dev/apcie2200_[0..15] input|output\n", argv[0]);
		exit(1);
	}

	if ( ( apcie2200_card_fd = open(argv[1], O_RDWR)) == -1 )
	{
		perror("open");
		exit(1);
	}

	{
		struct stat file_stat;
		if ( fstat(apcie2200_card_fd,&file_stat) )
		{
			perror("fstat");
			exit(1);
		}
		if (!S_ISCHR(file_stat.st_mode))
		{
			/* should never happen !*/
			printf("file %s is not a character device\n",argv[1]);
			printf("failed");
			exit(1);
		}
		printf("%s, major:%d minor:%d \n",argv[1], major(file_stat.st_rdev),minor(file_stat.st_rdev));
	}


	ucStopFlag=0;
	signal(SIGINT, v_ControlExit);

	/* run test */
	{
		printf( "\n++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"
			"\nRunning digital %s test on device %s (Ctrl+C to stop)"
			"\n++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n\n\n",
			ucTestInputs ? "inputs" : "outputs", argv[1]);
		
		if ( ucTestInputs )
		{
			apcie2200_test_digital_input (apcie2200_card_fd);
		}
		else
		{
			apcie2200_test_digital_output (apcie2200_card_fd);
		}
	}

	return 0;
}
//--------------------------------------------------------------------------------
