/** @file dig_outputs.c
 
   Contains all functions to set digital outputs of the APCIE-2200.
 
   @par CREATION  
   @author Krauth Julien
   @date   07.01.09
   
   @par VERSION
   @verbatim
   $LastChangedRevision:$
   $LastChangedDate:$
   @endverbatim     
    
   @par LICENCE
   @verbatim
   Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
        ADDI-DATA GmbH
        Airpark Business Center
        Airport Boulevard B210
        77836 Rheinm�nster
        Germany
        Tel: +49(0)7229/1847-0
        Fax: +49(0)7229/1847-200
        http://www.addi-data-com
        info@addi-data.com
        
   This program is free software; you can redistribute it and/or modify it under 
   the terms of the GNU General Public License as published by the Free Software 
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. 
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with 
   this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   You shoud also find the complete GPL in the COPYING file 
   accompanying this source code.
   @endverbatim   
 */

#include <apcie2200-kapi.h>
#include "apcie2200-private.h"

/**@def EXPORT_NO_SYMBOLS
 * Function in this file are not exported.
 */
EXPORT_NO_SYMBOLS;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,27)
	#define __user 
#endif


//------------------------------------------------------------------------------

/** Set all digital outputs of the APCIE-2200 on.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_Set8DigitalOutputsOn when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param [in]		arg		Function argument. An unsigned char that has contain port state to set.
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
int do_CMD_APCIE2200_Set16DigitalOutputsOn(struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	int ret;
      /* Test the value to set */
	if (arg > 65535 )
    	return -EINVAL;                        
 
 	{
 		unsigned long irqstate;
		if (APCIE2200_LOCK(pdev, &irqstate))
			return -ERESTARTSYS;
		{
			ret = APCIE2200_Set16DigitalOutputsOn(pdev, arg);
		}
		APCIE2200_UNLOCK(pdev,irqstate);
 	}
	return ret;
}

//------------------------------------------------------------------------------

/** Set all digital outputs of the APCIE-2200 off.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_Set8DigitalOutputsOn when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param [in]		arg		Function argument. An unsigned char that has contain port state to reset.
 *
 * @retval 0		No error. 
 * @retval -EPERM	Output memory is not activated.
 * @retval -EFAULT	Can't return parameters.
 */
int do_CMD_APCIE2200_Set16DigitalOutputsOff(struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	int ret;
 	{
 		unsigned long irqstate;
		if (APCIE2200_LOCK(pdev, &irqstate))
			return -ERESTARTSYS;
		{
			ret = APCIE2200_Set16DigitalOutputsOff(pdev, arg);
		}
		APCIE2200_UNLOCK(pdev,irqstate);
 	}
	return ret;
}

//------------------------------------------------------------------------------

/** Set the digital output memory on.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_SetDigitalOutputMemoryOn when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param 			arg		Non arg, it can be NULL.
 *
 * @retval 0		No error.
 */
int do_CMD_APCIE2200_SetDigitalOutputMemoryOn(struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	int ret;
	
	if (arg > 65535 )
    	return -EFAULT;
	
 	{
 		unsigned long irqstate;
		if (APCIE2200_LOCK(pdev, &irqstate))
			return -ERESTARTSYS;
		{
			ret = APCIE2200_SetDigitalOutputMemoryOn(pdev);
		}
		APCIE2200_UNLOCK(pdev,irqstate);
 	}
    return ret;
}

//------------------------------------------------------------------------------

/** Set the digital output memory off.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_SetDigitalOutputMemoryOff when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param 			arg		Non arg, it can be NULL.
 *
 * @retval 0		No error.
 */
int do_CMD_APCIE2200_SetDigitalOutputMemoryOff(struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	int ret;
 	{
 		unsigned long irqstate;
		if (APCIE2200_LOCK(pdev, &irqstate))
			return -ERESTARTSYS;
		{
			ret = APCIE2200_SetDigitalOutputMemoryOff(pdev);
		}
		APCIE2200_UNLOCK(pdev,irqstate);
 	}
    return ret;
}

//------------------------------------------------------------------------------

/** Get all digital outputs states of the APCIE-2200.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_Get8DigitalOutputStatus when using the IOCTL funktion.<br>
 * 							To call the function in kernel mode, cmd must be 0.
 * @param [out]		arg		Function argument. An unsigned char that contain port state.
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
int do_CMD_APCIE2200_Get16DigitalOutputStatus(struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	uint32_t PortState = 0;
	int ret;
 	{
 		unsigned long irqstate;
		if (APCIE2200_LOCK(pdev, &irqstate))
			return -ERESTARTSYS;
		{
			ret = APCIE2200_Get16DigitalOutputStatus(pdev, &PortState);
		}
		APCIE2200_UNLOCK(pdev,irqstate);
		
		if(ret) 
			return ret;
 
		/* Returns port state to the user */
		if ( copy_to_user( (unsigned int __user *) arg, (unsigned int*) &PortState, sizeof(unsigned int)) )
			return -EFAULT;       
 	}
	return 0;
}
	
//------------------------------------------------------------------------------

/** Get the number of digital outputs.
 *
 * @param [in,out]	pdev	Pointer on the device structure.
 * @param [in]		cmd		IOCTL command. The command is <br>
 * 							CMD_APCIE2200_GetNumberOfDigitalOutputs when using the IOCTL function.
 * @param [out]		arg		The number of digital outputs.<br>
 *
 * @retval 0		No error.
 * @retval -EFAULT	Can't get parameters.
 */
int do_CMD_APCIE2200_GetNumberOfDigitalOutputs (struct pci_dev *pdev, unsigned int cmd, unsigned long arg)
{
	int nbr = APCIE2200_GetNumberOfDigitalOutputs (pdev);

	if (copy_to_user ((void __user *) arg, &nbr , sizeof (uint32_t)))
	{
		return -EFAULT;
	}

	return 0;
}

//------------------------------------------------------------------------------

