 /* @file apcie2200.h
  * 
  * @brief Define the public API for the apcie2200 ADDI-DATA board.
  * 
  * @author Sylvain Nahas, Julien Krauth
  */

/** @par LICENCE
 * 
 *  @verbatim
   Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
   ADDI-DATA GmbH
   Airpark Business Center
   Airport Boulevard B210
   77836 Rheinmuenster
   Germany
   Tel: +49(0)7229/1847-0
   Fax: +49(0)7229/1847-200
   http://www.addi-data-com
   info@addi-data.com
 
 This library is free software; you can redistribute it and/or modify it under 
 the terms of the GNU Lesser General Public License as published by the 
 Free Software Foundation; either version 2.1 of the License, 
 or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful, 
 but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 or FITNESS FOR A PARTICULAR PURPOSE. 
 See the GNU Lesser General Public License for more details.
 
 You should have received a copy of the GNU Lesser General Public License 
 along with this library; if not, write to the Free Software Foundation, 
 Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 
 
 You also shoud find the complete LGPL in the LGPL.txt file accompanying 
 this source code.
 
 @endverbatim
**/

#ifndef __APCIE2200_H__
#define __APCIE2200_H__


#include <apci.h>

//GENERAL DEFINE

#define APCIE2200_BOARD_NAME                      "APCIE2200"

#define APCIE2200_BOARD_VENDOR_ID                 0x15B8

/* Digital inputs on the 50 pin connector (to use with a FB2200e cable) */
#define APCIE2200_16_16_BOARD_DEVICE_ID           0x1013
#define APCIE2200_16_8_BOARD_DEVICE_ID            0x1014
#define APCIE2200_8_8_BOARD_DEVICE_ID             0x1015

/* No digital inputs */
#define APCIE2200_8_BOARD_DEVICE_ID               0x1016
#define APCIE2200_16_BOARD_DEVICE_ID              0x1017

/**@def APCIE2200_READREG
 * Read register offset (channel 0 - 7).
 */
#define APCIE2200_READREG			0x00

/**@def APCIE2200_WRITE
 * Write register offset (channel 0 - 7).
 */
#define APCIE2200_WRITEREG			0x04

/* value used for CMD_APCIE2200_CheckAndGetPCISlotNumber ! DEPRECATED ! */
#ifndef CONFIG_APCIE2200_MAX_BOARD_NBR
#define CONFIG_APCIE2200_MAX_BOARD_NBR                   (3)
#endif // CONFIG_APCIE2200_MAX_BOARD_NBR

// for compatibility with older driver (deprecated)
#define APCIE2200_MAX_BOARD_NBR CONFIG_APCIE2200_MAX_BOARD_NBR

// IOCTL define part
#define APCIE2200_MAGIC (char) 'A'

/**
 * @param (char*) area of size CONFIG_APCIE2200_MAX_BOARD_NBR+1
 * 
 * @deprecated
 * 
 */
#define CMD_APCIE2200_CheckAndGetPCISlotNumber   _IOR(APCIE2200_MAGIC, 1,char*)

#define CMD_APCIE2200_SetBoardInformation        _IOW(APCIE2200_MAGIC, 2,USHORT)

/**
 * @deprecated
 */
#define CMD_APCIE2200_GetHardwareInformation     _IOR(APCIE2200_MAGIC, 3,str_BaseInformations*)

/** Read all digital inputs of the APCIE-2200.
 *
 * @param [out]		arg		Port state (0 to 65535).
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
#define CMD_APCIE2200_Read16DigitalInputs			_IOR(APCIE2200_MAGIC, 4, uint32_t *)

/** Set all digital outputs of the APCIE-2200 on.
 *
 * @param [in]		arg		Port state to set (0 to 65535).
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
#define CMD_APCIE2200_Set16DigitalOutputsOn		_IOW(APCIE2200_MAGIC, 5, uint32_t)

/** Set all digital outputs of the APCIE-2200 off.
 *
 * @param [in]		arg		Port state to reset (0 to 65535).
 *
 * @retval 0		No error. 
 * @retval -EPERM	Output memory is not activated.
 * @retval -EFAULT	Can't return parameters.
 */
#define CMD_APCIE2200_Set16DigitalOutputsOff		_IOW(APCIE2200_MAGIC, 6, uint32_t)

/** Set the digital output memory on.
 *
 * @param 			arg		No arg, it can be NULL.
 *
 * @retval 0		No error.
 */
#define CMD_APCIE2200_SetDigitalOutputMemoryOn	_IO(APCIE2200_MAGIC, 7)

/** Set the digital output memory off.
 *
 * @param 			arg		No arg, it can be NULL.
 *
 * @retval 0		No error.
 */
#define CMD_APCIE2200_SetDigitalOutputMemoryOff	_IO(APCIE2200_MAGIC, 8)

/** Get all digital outputs state of the APCIE-2200.
 *
 * @param [out]		arg		Port state (0 to 65535).
 *
 * @retval 0		No error. 
 * @retval -EFAULT	Can't return parameters.
 */
#define CMD_APCIE2200_Get16DigitalOutputStatus	_IOR(APCIE2200_MAGIC, 9, uint32_t *)

/** 
 * @param[in] arg	struct 
 * 					{
 * 						int InterruptLogic;
 * 						uint32_t InterruptMaskMode1;
 *						uint32_t InterruptMaskMode2;
 * 					}
 * 
 * @return 0 on success, -1 on failure (see errno then) 
 */
#define CMD_APCIE2200_InitAndStartDigitalInputInterrupt   _IOW(APCIE2200_MAGIC, 10,void*)

/** 
 * 
 * @param arg None ( use NULL )
 * 
 * @return 0 on success, -1 on failure (see errno then)
 *  
 */
#define CMD_APCIE2200_ClearAndStopDigitalInputInterrupt   _IOW(APCIE2200_MAGIC, 11,void*)

/**
 * @param[in] arg	struct 
 * 					{
 * 						int Enabled; 
 * 						int InterruptLogic;
 *						uint32_t InterruptMaskMode1;
 *						uint32_t InterruptMaskMode2;
 * 					}
 * 
 * @return 0 on success, -1 on failure (see errno then)
 *  
 */
#define CMD_APCIE2200_GetDigitalInputInterruptConfig   _IOW(APCIE2200_MAGIC, 12,void*)

/** Empty event FIFO of all pending events.
 * 
 * @param arg None (Use NULL)
 * 
 * @return 0 on success, -1 on failure (see errno then) 
 * 
 */
#define CMD_APCIE2200_EventEmpty         			_IOW(APCIE2200_MAGIC, 13,void*)

/** Wait for an event.
 * 
 * @param[in] arg apcie2200_event_t*
 * 
 * @return 0 on success, -1 on failure (see errno then) 
 * 
 */
#define CMD_APCIE2200_EventWait         			_IOW(APCIE2200_MAGIC, 14,apcie2200_event_t*)

/** Configure the given timer, then start it.
 *
 * @param [in] arg Pointer to a structure containing the initialisation parameters \n
 * struct
	{
		uint32_t ressourceType;
		uint32_t mode;
		uint32_t timeBaseUnit;
		uint32_t timeBaseValue;
		uint32_t irq;
	};
 * with:
 *   ressourceType : 0: Watchdog
 *					 1: Timer
 *
 *   TimerMode: At this time only mode 2 is available (value 2).
 *
 *   TimerUnit: This is the unit for the time base.
 *              us: 1
 * 				ms: 2
 * 				s:  3
 *
 *   ReloadValue: 	12 Bit for Watchdog.
 *   				16 Bit for Timer.
 *
 *   irq: 1: Enable interrupt
 *        0: Disable interrupt
 *
 * @retval 0: On success
 * @retval -1: On failure (see errno and check parameters).
 */
#define CMD_APCIE2200_TimerWatchdogInitAndStart			_IOW (APCIE2200_MAGIC, 15, void*)


/** Release the timer / watchdog
 *
 * @param [in] arg:  0: Watchdog
 *					 1: Timer
 *
 * @retval 0: On success
 * @retval -1: On failure (see errno and check parameters).
 */
#define CMD_APCIE2200_TimerWatchdogStopAndRelease			_IOW (APCIE2200_MAGIC, 16, uint32_t)

/** Trigger the timer / watchdog
 *
 * @param [in] arg:  0: Watchdog
 *					 1: Timer
 *
 * @retval 0: On success
 * @retval -1: On failure (see errno and check parameters).
 */
#define CMD_APCIE2200_TimerWatchdogTrigger			_IOW (APCIE2200_MAGIC, 17, uint32_t)

/** Read the given timer.
 *
 * @param [in] arg Pointer to a structure containing the parameters \n
 * struct
	{
		uint32_t ressourceType;
		uint32_t value;
	};
 * with:
 *   ressourceType : 0: Watchdog
 *					 1: Timer
 *
 *   value: The timer value.
 *
 * @retval 0: On success
 * @retval -1: On failure (see errno and check parameters).
 */
#define CMD_APCIE2200_TimerWatchdogRead				_IOW (APCIE2200_MAGIC, 18, void*)

/** Get status of the given timer.
 *
 * @param [in,out] arg Pointer to a structure containing the parameters \n
 * struct
	{
		uint32_t ressourceType;
		uint8_t overflow;
		uint8_t softTrigger;
	};
 * with:
 *   ressourceType : 0: Watchdog
 *					 1: Timer
 * 
 *   overflow: 1 when overflow.
 *   overflow: 1 when software trigger has been used.
 *
 * @retval 0: On success
 * @retval -1: On failure (see errno and check parameters).
 */
#define CMD_APCIE2200_TimerWatchdogGetStatus		_IOW (APCIE2200_MAGIC, 19, void*)

/** Get the number of digital outputs.
 *
 * @param [in,out]	arg	The number of digital outputs.
 *
 * @retval 0		No error.
 * @retval -EFAULT	Can't return parameter.
 */
#define CMD_APCIE2200_GetNumberOfDigitalOutputs 	_IOW (APCIE2200_MAGIC, 20, void*)

/** Get the number of digital inputs.
 *
 * @param [in,out]	arg	The number of digital inputs.
 *
 * @retval 0		No error.
 * @retval -EFAULT	Can't return parameter.
 */
#define CMD_APCIE2200_GetNumberOfDigitalInputs 	_IOW (APCIE2200_MAGIC, 21, void*)

/** Used internally. This is the ioctl CMD with the highest number.
 * @internal
 */
#define __APCIE2200_UPPER_IOCTL_CMD 21

#endif // __APCIE2200_H__
