/** @file sample_imp_cpt_01.c
*
* @author S. Weber
*
* Initialize and read 1 impuls counter value:
* - i_APCI1710_InitPulseEncoder
* - i_APCI1710_EnablePulseEncoder
* - i_APCI1710_DisablePulseEncoder
* - i_APCI1710_ReadPulseEncoderStatus
* - i_APCI1710_ReadPulseEncoderValue
*/

/** @par LICENCE
* @verbatim
    Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
    ADDI-DATA GmbH
    Airpark Business Center
    Airport Boulevard B210
    77836 Rheinm�nster
    Germany
    Tel: +49(0)7229/1847-0
    Fax: +49(0)7229/1847-200
    http://www.addi-data-com
    info@addi-data.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. 

    You shoud find the complete GPL in the COPYING file accompanying 
    this source code.
* @endverbatim
*/

#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
    #include <linux/config.h>
#else
    #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,33)
        #include <linux/autoconf.h>
    #else
        #include <generated/autoconf.h>
    #endif
#endif

#include <linux/spinlock.h>
#include <linux/ioctl.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/pci.h> // struct pci_dev
#include <asm/io.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,4,0)
    #include <asm/system.h>
#endif
#include <asm/uaccess.h>
#include <asm/bitops.h>
#include <linux/sched.h>

#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <apci1710.h>
#include <apci1710-kapi.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("ADDI-DATA GmbH <info@addi-data.com>");
MODULE_DESCRIPTION("APCI-1710");

EXPORT_NO_SYMBOLS;

static const char modulename[] = "sample_imp_cpt";


static int sample_imp_cpt_kernel (struct pci_dev *pdev)
{ 

    int i_ReturnValue = 0;
    int i = 0;
    unsigned long irqstate;

    uint8_t   b_ModulNbr            = 0;    /* The first module */
    uint8_t   b_PulseEncoderNbr     = 0;    /* The first impulse counter */
    uint8_t   b_InputLevelSelection = 1;    /* Set pulse encoder count the the high level pulse */
    uint8_t   b_TriggerOutputAction = 0;    /* No trigger action */
    uint32_t ul_StartValue          = 1000; /* Pulse encoder start, reload value */
    uint8_t   b_CycleSelection      = APCI1710_CONTINUOUS;
    uint8_t   b_InterruptHandling   = 0; /* Interrupts are disabled */
    uint8_t   b_Status              = 0;
    uint32_t ul_ReadValue           = 0;

    /* Lock the function to avoid parallel configurations */
    apci1710_lock(pdev,&irqstate);

    i_ReturnValue = i_APCI1710_InitPulseEncoder (pdev,
                                                     b_ModulNbr,
                                                     b_PulseEncoderNbr,
                                                     b_InputLevelSelection,
                                                     b_TriggerOutputAction,
                                                     ul_StartValue);

    /* Unlock the function so that other applications can call it */
    apci1710_unlock(pdev,irqstate);

    switch (i_ReturnValue)
    {
        case 0:
        printk ("\ni_APCI1710_InitPulseEncoder OK");
        break;

        case 1:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
        break;
        
        case 2:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. The module is not a pulse encoder module", i_ReturnValue);
        break;
        
        case 3:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
        break;
        
        case 4:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. Input level selection is wrong", i_ReturnValue);
        break;
        
        case 5:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. Digital TRIGGER output action selection is wrong", i_ReturnValue);
        break;
        
        case 6:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d. Pulse encoder start value is wrong", i_ReturnValue);
        break;
        
        
        default:
        printk ("\ni_APCI1710_InitPulseEncoder error");
        printk ("\nError = %d.", i_ReturnValue);
        break;
    }


    /* Lock the function to avoid parallel configurations */
    apci1710_lock(pdev,&irqstate);

    i_ReturnValue = i_APCI1710_EnablePulseEncoder (pdev,
                                                       b_ModulNbr,
                                                       b_PulseEncoderNbr,
                                                       b_CycleSelection,
                                                       b_InterruptHandling);

    /* Unlock the function so that other applications can call it */
    apci1710_unlock(pdev,irqstate);

    switch (i_ReturnValue)
    {
        case 0:
        printk ("\ni_APCI1710_EnablePulseEncoder OK");
        break;

        case 1:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
        break;
        
        case 2:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Module selection is wrong", i_ReturnValue);
        break;
        
        case 3:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
        break;
        
        case 4:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Pulse encoder not initialised", i_ReturnValue);
        break;
        
        case 5:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Cycle selection mode is wrong", i_ReturnValue);
        break;
        
        case 6:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Interrupt handling mode is wrong", i_ReturnValue);
        break;
        
        case 7:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d. Interrupt routine not installed", i_ReturnValue);
        break;
        
        default:
        printk ("\ni_APCI1710_EnablePulseEncoder error");
        printk ("\nError = %d.", i_ReturnValue);
        break;
    }

    if (!i_ReturnValue)
    {
        for (i=0; i<100; i++)
        {

            /* Lock the function to avoid parallel configurations */
            apci1710_lock(pdev,&irqstate);

            i_ReturnValue = i_APCI1710_ReadPulseEncoderStatus (pdev,
                                                                           b_ModulNbr,
                                                                           b_PulseEncoderNbr,
                                                                           &b_Status);

            /* Unlock the function so that other applications can call it */
            apci1710_unlock(pdev,irqstate);
        
            switch (i_ReturnValue)
            {
                case 0 :
                    /* Lock the function to avoid parallel configurations */
                    apci1710_lock(pdev,&irqstate);

                    i_ReturnValue = i_APCI1710_ReadPulseEncoderValue (pdev,
                                                                          b_ModulNbr,
                                                              b_PulseEncoderNbr,
                                                          &ul_ReadValue);

                    /* Unlock the function so that other applications can call it */
                    apci1710_unlock(pdev,irqstate);
        
                    switch (i_ReturnValue)
                    {
                        case 0 :
                            printk ("\n%i) Counter value = %10u Overflow = %1u", i, ul_ReadValue, b_Status);
                            break;
                
                        case 1:
                            printk ("\ni_APCI1710_ReadPulseEncoderValue error");
                            printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                            break;
                
                        case 2:
                            printk ("\ni_APCI1710_ReadPulseEncoderValue error");
                            printk ("\nError = %d. The module parameter is wrong", i_ReturnValue);
                            break;
                
                        case 3:
                            printk ("\ni_APCI1710_ReadPulseEncoderValue error");
                            printk ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
                            break;
                
                        case 4:
                            printk ("\ni_APCI1710_ReadPulseEncoderValue error");
                            printk ("\nError = %d. Pulse encoder not initialised", i_ReturnValue);
                            break;
                        
                        default:
                            printk ("\ni_APCI1710_ReadPulseEncoderValue error");
                            printk ("\nError = %d.", i_ReturnValue);
                            break;
                    }
                    break;
                
                case 1:
                    printk ("\ni_APCI1710_ReadPulseEncoderStatus error");
                    printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                    break;
                
                case 2:
                    printk ("\ni_APCI1710_ReadPulseEncoderStatus error");
                    printk ("\nError = %d. The module parameter is wrong", i_ReturnValue);
                    break;
                
                case 3:
                    printk ("\ni_APCI1710_ReadPulseEncoderStatus error");
                    printk ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
                    break;
                
                case 4:
                    printk ("\ni_APCI1710_ReadPulseEncoderStatus error");
                    printk ("\nError = %d. Pulse encoder not initialised", i_ReturnValue);
                    break;
                default:
                    printk ("\ni_APCI1710_ReadPulseEncoderStatus error");
                    printk ("\nError = %d.", i_ReturnValue);
                    break;
            }
        }
    }

    /* Lock the function to avoid parallel configurations */
    apci1710_lock(pdev,&irqstate);

    i_ReturnValue = i_APCI1710_DisablePulseEncoder (pdev,
                                                        b_ModulNbr,
                                                        b_PulseEncoderNbr);

    /* Unlock the function so that other applications can call it */
    apci1710_unlock(pdev,irqstate);

    switch (i_ReturnValue)
    {
        case 0:
        printk ("\ni_APCI1710_DisablePulseEncoder OK");
        break;

        case 1:
        printk ("\ni_APCI1710_DisablePulseEncoder error");
        printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
        break;
        
        case 2:
        printk ("\ni_APCI1710_DisablePulseEncoder error");
        printk ("\nError = %d. Module selection is wrong", i_ReturnValue);
        break;
        
        case 3:
        printk ("\ni_APCI1710_DisablePulseEncoder error");
        printk ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
        break;
        
        case 4:
        printk ("\ni_APCI1710_DisablePulseEncoder error");
        printk ("\nError = %d. Pulse encoder not initialised", i_ReturnValue);
        break;
        
        default:
        printk ("\ni_APCI1710_DisablePulseEncoder error");
        printk ("\nError = %d.", i_ReturnValue);
        break;
    }
    
    return (i_ReturnValue);
}


/** Called when module loads. */
static int __init apci1710Sample_init(void)
{
    struct pci_dev *pdev;
    int board_index = 0;

    printk("%s: looking for board %u\n",modulename,board_index);
    pdev=apci1710_lookup_board_by_index(0);
    if( !pdev )
    {
        printk("%s: board %u not found\n",modulename,board_index);
        return -ENODEV;
    }
    printk("%s: beginning test\n",modulename);
        sample_imp_cpt_kernel (pdev);

    return 0;
}
//-------------------------------------------------------------------
/** Called when module is unloaded. */
static void __exit apci1710Sample_exit(void)
{
 
    
}
//------------------------------------------------------------------------------
module_exit(apci1710Sample_exit);
module_init(apci1710Sample_init);
