/** @file sample_inc_counter_01.c
*
* @author Krauth Julien
*
* Initialize and read 1 incremental counter.
* - i_APCI1710_InitCounter
* - i_APCI1710_ClearCounterValue
* - i_APCI1710_Read32BitCounterValue
*/

/** @par LICENCE
* @verbatim
    Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
    ADDI-DATA GmbH
    Airpark Business Center
    Airport Boulevard B210
    77836 Rheinm�nster
    Germany
    Tel: +49(0)7229/1847-0
    Fax: +49(0)7229/1847-200
    http://www.addi-data-com
    info@addi-data.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. 

    You shoud find the complete GPL in the COPYING file accompanying 
    this source code.
* @endverbatim
*/

#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
    #include <linux/config.h>
#else
    #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,33)
        #include <linux/autoconf.h>
    #else
        #include <generated/autoconf.h>
    #endif
#endif

#include <linux/spinlock.h>
#include <linux/ioctl.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/pci.h> // struct pci_dev
#include <asm/io.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,4,0)
    #include <asm/system.h>
#endif
#include <asm/uaccess.h>
#include <asm/bitops.h>
#include <linux/sched.h>

#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <apci1710.h>
#include <apci1710-kapi.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("ADDI-DATA GmbH <info@addi-data.com>");
MODULE_DESCRIPTION("APCI-1710");

EXPORT_NO_SYMBOLS;

static const char modulename[] = "sample_incremental_counter";


static int sample_inc_counter_kernel (struct pci_dev *pdev)
{ 

    int i_ReturnValue = 0;
    int i = 0;
    unsigned long irqstate;

    uint8_t b_ModulNbr = 0;                                 // The module to use (0 to 3).
    uint8_t b_CounterRange = APCI1710_32BIT_COUNTER;        // Selection form counter range.
    uint8_t b_FirstCounterModus = APCI1710_SIMPLE_MODE;     // First counter operating mode.
    uint8_t b_FirstCounterOption = APCI1710_HYSTERESIS_ON;  // First counter option.
    uint8_t b_SecondCounterModus = APCI1710_SIMPLE_MODE;    // Second counter operating mode.
    uint8_t b_SecondCounterOption = APCI1710_HYSTERESIS_ON; // Second counter option.
    uint32_t ul_CounterValue = 0;
    
    /* Lock the function to avoid parallel configurations */
    apci1710_lock(pdev,&irqstate);

    /* Initialise the incremental counter */
    i_ReturnValue = i_APCI1710_InitCounter (pdev,
                                            b_ModulNbr,
                                            b_CounterRange,
                                            b_FirstCounterModus,
                                            b_FirstCounterOption,
                                            b_SecondCounterModus,
                                            b_SecondCounterOption);

    /* Unlock the function so that other applications can call it */
    apci1710_unlock(pdev,irqstate);

    switch (i_ReturnValue)
    {
        case 0:
        {
            printk ("\ni_APCI1710_InitCounter OK");

            /* Lock the function to avoid parallel configurations */
            apci1710_lock(pdev,&irqstate);
                        
            i_ReturnValue = i_APCI1710_ClearCounterValue (pdev, b_ModulNbr);
    
            /* Unlock the function so that other applications can call it */
            apci1710_unlock(pdev,irqstate);
                
            switch (i_ReturnValue)
            {
                case 0:
                    printk ("\ni_APCI1710_ClearCounterValue OK");
                break;

                case 1:
                    printk ("\ni_APCI1710_ClearCounterValue error");
                    printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                    return 0;                        
                break;

                case 2:
                    printk ("\ni_APCI1710_ClearCounterValue error");
                    printk ("\nError = %d. The selected module number parameter is wrong", i_ReturnValue);
                    return 0;                        
                break;

                case 3:
                    printk ("\ni_APCI1710_ClearCounterValue error");
                    printk ("\nError = %d. Counter not initialised", i_ReturnValue);
                    return 0;                        
                break;

                default:
                    printk ("\ni_APCI1710_ClearCounterValue error");
                    printk ("\nError = %d.", i_ReturnValue);
                    return 0;
                break;
            }

            printk ("\n\n");
            
            do
            {
                /* Read the 32-Bit counter */    
                i_ReturnValue = i_APCI1710_Read32BitCounterValue (pdev,
                                                                  b_ModulNbr,
                                                                  &ul_CounterValue);

                /* Test the return value */
                switch (i_ReturnValue)
                {
                    case 0:
                    break;

                    case 1:
                        printk ("\ni_Read32BitCounterValue for the first counter error");
                        printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                        return 0;                            
                    break;

                    case 2:
                        printk ("\ni_Read32BitCounterValue for the first counter error");
                        printk ("\nError = %d. No counter module found", i_ReturnValue);
                        return 0;                            
                    break;

                    case 3:
                        printk ("\ni_Read32BitCounterValue for the first counter error");
                        printk ("\nError = %d. Counter not initialised", i_ReturnValue);
                        return 0;
                    break;

                    default:
                        printk ("\ni_Read32BitCounterValue for the first counter error");
                        printk ("\nError = %d.", i_ReturnValue);
                        return 0;                            
                    break;
                }

                printk ("\nCounter value from module %d  = %u", b_ModulNbr, ul_CounterValue);
            }
            while (i++ < 100);    
            
        }    
        break;
                
        case 1:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
        break;

        case 2:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The module is not a counter module", i_ReturnValue);
        break;

        case 3:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The selected counter range is wrong", i_ReturnValue);
        break;

        case 4:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The selected first counter operating mode is wrong", i_ReturnValue);
        break;

        case 5:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The selected first counter operating option is wrong", i_ReturnValue);
        break;

        case 6:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The selected second counter operating mode is wrong", i_ReturnValue);
        break;

        case 7:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d. The selected second counter operating option is wrong", i_ReturnValue);
        break;

        default:
            printk ("\ni_APCI1710_InitCounter error");
            printk ("\nError = %d", i_ReturnValue);    
        break;
    }

    return 0;
}


/** Called when module loads. */
static int __init apci1710Sample_init(void)
{
    struct pci_dev *pdev;
    int board_index = 0;

    printk("%s: looking for board %u\n",modulename,board_index);
    pdev=apci1710_lookup_board_by_index(0);
    if( !pdev )
    {
        printk("%s: board %u not found\n",modulename,board_index);
        return -ENODEV;
    }
    printk("%s: beginning test\n",modulename);
    sample_inc_counter_kernel (pdev);

    return 0;
}
//-------------------------------------------------------------------
/** Called when module is unloaded. */
static void __exit apci1710Sample_exit(void)
{
 
    
}
//------------------------------------------------------------------------------
module_exit(apci1710Sample_exit);
module_init(apci1710Sample_init);
