/*
  +-----------------------------------------------------------------------+
  | (C) ADDI-DATA GmbH
  | Project   : API APCI1710      |     Compiler   : BORLANDC/MICROSOFT C |
  | Module name : PWM.C           |     Version    : 3.1     / 6.0        |
  +-------------------------------+---------------------------------------+
  | Author    : Pravin         |     Date       : 22.01.99                |
  +-----------------------------------------------------------------------+
  | Description :   APCI-1710 Pulse wide modulation module                |

*/


#include "apci1710-private.h"

EXPORT_SYMBOL(i_APCI1710_InitPWM);
EXPORT_SYMBOL(i_APCI1710_SetNewPWMTiming);
EXPORT_SYMBOL(i_APCI1710_DisablePWM);
EXPORT_SYMBOL(i_APCI1710_EnablePWM);
EXPORT_SYMBOL(i_APCI1710_GetPWMInitialisation);
EXPORT_SYMBOL(i_APCI1710_GetPWMStatus);
EXPORT_SYMBOL(i_APCI1710_ReadPWM1DigitalInput );
EXPORT_SYMBOL(i_APCI1710_ReadPWMAllDigitalInput);
EXPORT_SYMBOL(i_APCI1710_EnableDisablePWMDigitalOutputManualSetting);
EXPORT_SYMBOL(i_APCI1710_SetPWMDigitalOutputOn);
EXPORT_SYMBOL(i_APCI1710_SetPWMDigitalOutputOff);

EXPORT_NO_SYMBOLS;
/*
+----------------------------------------------------------------------------+
| Function Name     : int i_APCI1710_InitPWM                               |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr,               |
|                                        uint8_t_     b_PWM,                    |
|                                        uint8_t_     b_ClockSelection,         |
|                                        uint8_t_     b_TimingUnit,             |
|                                        uint32_t   ul_LowTiming,              |
|                                        uint32_t   ul_HighTiming)         |
+----------------------------------------------------------------------------+
| Task              : Configure the selected PWM (b_PWM) from selected module|
|                     (b_ModulNbr). The ul_LowTiming, ul_HighTiming and      |
|                     ul_TimingUnit determine the low/high timing base for   |
|                     the period. pul_RealLowTiming, pul_RealHighTiming      |
|                     return the real timing value.                          |
|                     You must calling this function be for you call any     |
|                     other function witch access of the PWM.                |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_     b_BoardHandle    : Handle of board APCI-1710 |
|                     uint8_t_     b_ModulNbr       : Module number to configure|
|                                                  (0 to 3)                  |
|                     uint8_t_     b_PWM            : Selected PWM (0 or 1).    |
|                     uint8_t_     b_ClockSelection : Selection from PCI bus    |
|                                                  clock                     |
|                                                   - APCI1710_30MHZ :       |
|                                                     The PC have a 30 MHz   |
|                                                     PCI bus clock          |
|                                                   - APCI1710_33MHZ :       |
|                                                     The PC have a 33 MHz   |
|                                                     PCI bus clock          |
|                                                   - APCI1710_40MHZ         |
|                                                     The APCI-1710 have a   |
|                                                     integrated 40Mhz       |
|                                                     quartz.                |
|                     uint8_t_     b_TimingUnit     : Base timing Unit (0 to 4) |
|                                                       0 : ns               |
|                                                       1 : �s               |
|                                                       2 : ms               |
|                                                       3 : s                |
|                                                       4 : mn               |
|                     uint32_t    ul_LowTiming     : Low base timing value.    |
|                     uint32_t    ul_HighTiming    : High base timing value.   |
+----------------------------------------------------------------------------+
| Output Parameters :														 |
+----------------------------------------------------------------------------+
| Return Value     : 0: No error                                            |
|                    1: Module selection wrong                              |
|                    2: The module is not a PWM module                      |
|                    3: PWM selection is wrong                              |
|                    4: The selected input clock is wrong                   |
|                    5: Timing Unit selection is wrong                      |
|                    6: Low base timing selection is wrong                  |
|                    7: High base timing selection is wrong                 |
|                    8: You can not use the 40MHz clock selection with     |
|                        this board                                          |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_InitPWM      (struct pci_dev *pdev,
		uint8_t     b_ModulNbr,
		uint8_t     b_PWM,
		uint8_t     b_ClockSelection,
		uint8_t     b_TimingUnit,
		uint32_t   ul_LowTiming,
		uint32_t   ul_HighTiming)
	{
	int    i_ReturnValue = 0;
	uint32_t ul_LowTimerValue;
	uint32_t ul_HighTimerValue;
	uint32_t dw_Command;
	uint32_t d_ClockSelection = 0;


	/**************************/
	/* Test the module number */
	/**************************/

	if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
	   {
	   /***************/
	   /* Test if PWM */
	   /***************/

	   if ( APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
	      /**************************/
	      /* Test the PWM selection */
	      /**************************/

	      if (b_PWM >= 0 && b_PWM <= 1)
		 {
		 /******************/
		 /* Test the clock */
		 /******************/

		 if ((b_ClockSelection == APCI1710_30MHZ)  ||
		     (b_ClockSelection == APCI1710_33MHZ)  ||
		     (b_ClockSelection == APCI1710_40MHZ))
		    {
		    /************************/
		    /* Test the timing unit */
		    /************************/

		    if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		       {
		       /*********************************/
		       /* Test the low timing selection */
		       /*********************************/

		       if (((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 200) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571230650UL))  ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571230UL))     ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571UL))        ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 9UL))          ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 182) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 519691043UL))  ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 519691UL))     ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 520UL))        ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 8UL))          ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 250) && (ul_LowTiming <= 0xFFFFFFFFUL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 150) && (ul_LowTiming <= 0xFFFFFFFFUL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429496729UL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429496UL))     ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429UL))        ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 7UL)))
			  {
			  /**********************************/
			  /* Test the High timing selection */
			  /**********************************/

			  if (((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 200) && (ul_HighTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571230650UL))  ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571230UL))     ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571UL))        ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 9UL))          ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 182) && (ul_HighTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 519691043UL))  ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 519691UL))     ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 520UL))        ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 8UL))          ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 250) && (ul_HighTiming <= 0xFFFFFFFFUL) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 150) && (ul_HighTiming <= 0xFFFFFFFFUL) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429496729UL))  ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429496UL))     ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429UL))        ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 7UL)))
			     {

					 if ((b_ClockSelection == APCI1710_30MHZ)  ||
					     (b_ClockSelection == APCI1710_33MHZ)  ||
					     (b_ClockSelection == APCI1710_40MHZ))
					    {
				/****************************************************/
				/* Time = ((4 * registerValue) + 6) * (1 / BaseClk) */
				/* Time * BaseClk = (4 * registerValue) + 6         */
				/* (Time * BaseClk) - 6 = 4 * registerValue         */
				/* ((Time * BaseClk) - 6) / 4 = registerValue       */
				/****************************************************/

				   /******/
				   /* ns */
				   /******/

				   if (b_TimingUnit == 0)
				      {
				      d_ClockSelection =  b_ClockSelection / 1000;
				      }

			           /******/
				   /* us */
				   /******/

				   if (b_TimingUnit == 1)
				      {
				      d_ClockSelection =  b_ClockSelection;
				      }

				   /******/
				   /* ms */
				   /******/

				   if (b_TimingUnit == 2)
				      {
				      d_ClockSelection =  b_ClockSelection * 1000;
				      }

				   /*****/
				   /* s */
				   /*****/

				   if (b_TimingUnit == 3)
				      {
				      d_ClockSelection =  b_ClockSelection * 1000000;
				      }

				   /******/
				   /* mn */
				   /******/

				   if (b_TimingUnit == 4)
				      {
				      d_ClockSelection =  b_ClockSelection * 60* 1000000;
				      }

				   /************************************/
				   /* Calculate the low division fator */
				   /************************************/

				   ul_LowTimerValue   = (uint32_t) ((( ul_LowTiming * d_ClockSelection) - 6) / 4);

				   /*************************************/
				   /* Calculate the high division fator */
				   /*************************************/

				   ul_HighTimerValue   = (uint32_t) ((( ul_HighTiming * d_ClockSelection) - 6) / 4);

				/****************************/
				/* Save the clock selection */
				/****************************/

				   APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
				s_PWMModuleInfo.
				b_ClockSelection = b_ClockSelection;

				/************************/
				/* Save the timing unit */
				/************************/

				   APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
				s_PWMModuleInfo.
				s_PWMInfo [b_PWM].
				b_TimingUnit = b_TimingUnit;

				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
				s_PWMModuleInfo.
				s_PWMInfo [b_PWM].
				ul_RealLowTiming = ul_LowTimerValue;

				/****************************/
				/* Save the high base timing */
				/****************************/

				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
				s_PWMModuleInfo.
				s_PWMInfo [b_PWM].
				ul_RealHighTiming = ul_HighTimerValue;
				/************************/
				/* Write the low timing */
				/************************/

				OUTPDW (GET_BAR2(pdev),
					((20 * b_PWM) +  MODULE_OFFSET(b_ModulNbr)), ul_LowTimerValue);

				/*************************/
				/* Write the high timing */
				/*************************/


				OUTPDW (GET_BAR2(pdev),
					(4 + (20 * b_PWM) +  MODULE_OFFSET(b_ModulNbr)), ul_HighTimerValue);

				/***************************/
				/* Set the clock selection */
				/***************************/

				INPDW (GET_BAR2(pdev),
				       (8 + (20 * b_PWM) +  MODULE_OFFSET(b_ModulNbr)), &dw_Command);

				dw_Command = dw_Command & 0x7F;

				if (b_ClockSelection == APCI1710_40MHZ)
				   {
				   dw_Command = dw_Command | 0x80;
				   }

				/***************************/
				/* Set the clock selection */
				/***************************/

				OUTPDW (GET_BAR2(pdev),
					(8 + (20 * b_PWM) +  MODULE_OFFSET(b_ModulNbr)), dw_Command);

				/*************/
				/* PWM init. */
				/*************/

				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
				s_PWMModuleInfo.
				s_PWMInfo [b_PWM].
				b_PWMInit = 1;
				}
			     else
				{
				/***************************************************/
				/* You can not use the 40MHz clock selection with */
				/* this board                                      */
				/***************************************************/

				i_ReturnValue = 8;
				}
			     }
			  else
			     {
			     /***************************************/
			     /* High base timing selection is wrong */
			     /***************************************/

			     i_ReturnValue = 7;
			     }
			  }
		       else
			  {
			  /**************************************/
			  /* Low base timing selection is wrong */
			  /**************************************/

			  i_ReturnValue = 6;
			  }
		       } // if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		    else
		       {
		       /**********************************/
		       /* Timing unit selection is wrong */
		       /**********************************/

		       i_ReturnValue = 5;
		       } // if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		    } // if ((b_ClockSelection == APCI1710_30MHZ) || (b_ClockSelection == APCI1710_33MHZ) || (b_ClockSelection == APCI1710_40MHZ))
		 else
		    {
		    /*******************************/
		    /* The selected clock is wrong */
		    /*******************************/

		    i_ReturnValue = 4;
		    } // if ((b_ClockSelection == APCI1710_30MHZ) || (b_ClockSelection == APCI1710_33MHZ) || (b_ClockSelection == APCI1710_40MHZ))
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int i_APCI1710_SetNewPWMTiming                           |
|                                       (uint8_t_     b_BoardHandle,           |
|                                        uint8_t_     b_ModulNbr,              |
|                                        uint8_t_     b_PWM,                   |
|                                        uint8_t_     b_ClockSelection,        |
|                                        uint8_t_     b_TimingUnit,            |
|                                        uint32_t   ul_LowTiming,              |
|                                        uint32_t   ul_HighTiming)             |
+----------------------------------------------------------------------------+
| Task              : Set a new timing. The ul_LowTiming, ul_HighTiming and  |
|                     ul_TimingUnit determine the low/high timing base for   |
|                     the period.                                            |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_     b_BoardHandle    : Handle of board APCI-1710 |
|                     uint8_t_     b_ModulNbr       : Module number to configure|
|                                                  (0 to 3)                  |
|                     uint8_t_     b_PWM            : Selected PWM (0 or 1).    |
|                     uint8_t_     b_TimingUnit     : Base timing Unit (0 to 4) |
|                                                       0 : ns               |
|                                                       1 : �s               |
|                                                       2 : ms               |
|                                                       3 : s                |
|                                                       4 : mn               |
|                     uint32_t    ul_LowTiming     : Low base timing value.    |
|                     uint32_t    ul_HighTiming    : High base timing value.   |
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value     : 0: No error                                            |
|                    1: Module selection wrong                              |
|                    2: The module is not a PWM module                      |
|                    3: PWM selection is wrong                              |
|                    4: PWM not initialised                                 |
|                    5: Timing Unit selection is wrong                      |
|                    6: Low base timing selection is wrong                  |
|                    7: High base timing selection is wrong                 |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_SetNewPWMTiming	(struct pci_dev *pdev,
		uint8_t     b_ModulNbr,
		uint8_t     b_PWM,
		uint8_t     b_TimingUnit,
		uint32_t   ul_LowTiming,
		uint32_t   ul_HighTiming)
	{
	uint8_t   b_ClockSelection;
	int    i_ReturnValue = 0;
	uint32_t ul_LowTimerValue;
	uint32_t ul_HighTimerValue;
	uint32_t ul_RealLowTiming;
	uint32_t ul_RealHighTiming;
	uint32_t dw_Status;
	uint32_t dw_Command;
	uint32_t d_ClockSelection = 0;
        
        printk("b_TimingUnit %d\n",b_TimingUnit);

	/**************************/
	/* Test the module number */
	/**************************/

	if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
	   {
	   /***************/
	   /* Test if PWM */
	   /***************/

		if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
	      /**************************/
	      /* Test the PWM selection */
	      /**************************/

	      if (b_PWM >= 0 && b_PWM <= 1)
		 {
		 /***************************/
		 /* Test if PWM initialised */
		 /***************************/

		 INPDW (GET_BAR2(pdev),
			(12 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
			 &dw_Status);

		 if (dw_Status & 0x10)
		    {
		    b_ClockSelection = APCI1710_PRIVDATA(pdev)->
					s_ModuleInfo [(int)b_ModulNbr].
				       s_PWMModuleInfo.
				       b_ClockSelection;

		    /************************/
		    /* Test the timing unit */
		    /************************/

		    if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		       {
		       /*********************************/
		       /* Test the low timing selection */
		       /*********************************/

		       if (((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 200) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571230650UL))  ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571230UL))     ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 571UL))        ||
			   ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 9UL))          ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 182) && (ul_LowTiming <= 0xFFFFFFFFUL)) ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 519691043UL))  ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 519691UL))     ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 520UL))        ||
			   ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 8UL))          ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 250) && (ul_LowTiming <= 0xFFFFFFFFUL) && (ul_LowTiming <= 0xFFFFFFFFUL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_LowTiming >= 150) && (ul_LowTiming <= 0xFFFFFFFFUL) && (ul_LowTiming <= 0xFFFFFFFFUL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 1) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429496729UL))  ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 2) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429496UL))     ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 3) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 429UL))        ||
			   ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 4) && (ul_LowTiming >= 1)   && (ul_LowTiming <= 7UL)))
			  {
			  /**********************************/
			  /* Test the High timing selection */
			  /**********************************/

			  if (((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 200) && (ul_HighTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571230650UL))  ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571230UL))     ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 571UL))        ||
			      ((b_ClockSelection == APCI1710_30MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 9UL))          ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 181) && (ul_HighTiming <= 0xFFFFFFFFUL)) ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 519691043UL))  ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 519691UL))     ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 520UL))        ||
			      ((b_ClockSelection == APCI1710_33MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 8UL))          ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 250) && (ul_HighTiming <= 0xFFFFFFFFUL))  ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 0) && (ul_HighTiming >= 150) && (ul_HighTiming <= 0xFFFFFFFFUL))  ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 1) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429496729UL))  ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 2) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429496UL))     ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 3) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 429UL))        ||
			      ((b_ClockSelection == APCI1710_40MHZ) && (b_TimingUnit == 4) && (ul_HighTiming >= 1)   && (ul_HighTiming <= 7UL)))
			     {
			     /****************************************************/
			     /* Time = ((4 * registerValue) + 6) * (1 / BaseClk) */
			     /* Time * BaseClk = (4 * registerValue) + 6         */
			     /* (Time * BaseClk) - 6 = 4 * registerValue         */
			     /* ((Time * BaseClk) - 6) / 4 = registerValue       */
			     /****************************************************/

				/******/
				/* ns */
				/******/

				if (b_TimingUnit == 0)
				   {
				   d_ClockSelection =  b_ClockSelection / 1000;
				   }

			        /******/
				/* �s */
				/******/

				if (b_TimingUnit == 1)
				   {
				   d_ClockSelection =  b_ClockSelection;
				   }

				/******/
				/* ms */
				/******/

				if (b_TimingUnit == 2)
				   {
				   d_ClockSelection =  b_ClockSelection * 1000;
				   }

				/*****/
				/* s */
				/*****/

				if (b_TimingUnit == 3)
				   {
				   d_ClockSelection =  b_ClockSelection * 1000000;
				   }

				/******/
				/* mn */
				/******/

				if (b_TimingUnit == 4)
				   {
				   d_ClockSelection =  b_ClockSelection * 60 * 1000000;
				   }

				/************************************/
				/* Calculate the low division fator */
				/************************************/

				ul_LowTimerValue = (uint32_t) ((( ul_LowTiming * d_ClockSelection) - 6) / 4);
				ul_RealLowTiming = (uint32_t) (((4 *  ul_LowTimerValue) + 6) / d_ClockSelection);


				/*************************************/
				/* Calculate the high division fator */
				/*************************************/

				ul_HighTimerValue = (uint32_t) ((( ul_HighTiming * d_ClockSelection) - 6) / 4);
				ul_RealHighTiming = (uint32_t) (((4 *  ul_HighTimerValue) + 6) / d_ClockSelection);


			     /************************/
			     /* Save the timing unit */
			     /************************/

				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
			     s_PWMModuleInfo.
			     s_PWMInfo [b_PWM].
			     b_TimingUnit = b_TimingUnit;

			     /****************************/
			     /* Save the low base timing */
			     /****************************/

				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
			     s_PWMModuleInfo.
			     s_PWMInfo [b_PWM].
			     ul_RealLowTiming = ul_RealLowTiming;

			     /****************************/
			     /* Save the high base timing */
			     /****************************/
				APCI1710_PRIVDATA(pdev)->
				s_ModuleInfo [(int)b_ModulNbr].
			     s_PWMModuleInfo.
			     s_PWMInfo [b_PWM].
			     ul_RealHighTiming = ul_RealHighTiming;

			     /************************/
			     /* Write the low timing */
			     /************************/

			     OUTPDW (GET_BAR2(pdev),
			    		 0 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
					ul_LowTimerValue);

			     /*************************/
			     /* Write the high timing */
			     /*************************/

			     OUTPDW (GET_BAR2(pdev),
			    		 4 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
						 ul_HighTimerValue);

			     /***************************/
			     /* Set the clock selection */
			     /***************************/

			     INPDW (GET_BAR2(pdev),
			    		 8 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
						 &dw_Command);

			     dw_Command = dw_Command & 0x7F;

			     if (b_ClockSelection == APCI1710_40MHZ)
				{
				dw_Command = dw_Command | 0x80;
				}

			     /***************************/
			     /* Set the clock selection */
			     /***************************/

    		     OUTPDW (GET_BAR2(pdev),
			    		 8 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
						 dw_Command);
			     }
			  else
			     {
			     /***************************************/
			     /* High base timing selection is wrong */
			     /***************************************/

			     i_ReturnValue = 7;
			     }
			  }
		       else
			  {
			  /**************************************/
			  /* Low base timing selection is wrong */
			  /**************************************/

			  i_ReturnValue = 6;
			  }
		       } // if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		    else
		       {
		       /**********************************/
		       /* Timing unit selection is wrong */
		       /**********************************/

		       i_ReturnValue = 5;
		       } // if ((b_TimingUnit >= 0) && (b_TimingUnit <= 4))
		    } // if (dw_Status & 0x10)
		 else
		    {
		    /***********************/
		    /* PWM not initialised */
		    /***********************/

		    i_ReturnValue = 4;
		    } // if (dw_Status & 0x10)
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int     i_APCI1710_EnablePWM                         |
|                                       (uint8_t_  b_BoardHandle,               |
|                                        uint8_t_  b_ModulNbr,                  |
|                                        uint8_t_  b_PWM,                       |
|                                        uint8_t_  b_StartLevel,                |
|                                        uint8_t_  b_StopMode,                  |
|                                        uint8_t_  b_StopLevel,                 |
|                                        uint8_t_  b_ExternGate,                |
|                                        uint8_t_  b_InterruptEnable)           |
+----------------------------------------------------------------------------+
| Task              : Enable the selected PWM (b_PWM) from selected module   |
|                     (b_ModulNbr). You must calling the "i_APCI1710_InitPWM"|
|                     function be for you call this function.                |
|                     If you enable the PWM interrupt, the PWM generate a    |
|                     interrupt after each period.                           |
|                     See function "i_APCI1710_SetBoardIntRoutineX" and the  |
|                     Interrupt mask description chapter.                    |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle     : Handle of board APCI-1710    |
|                     uint8_t_ b_ModulNbr        : Selected module number       |
|                                               (0 to 3)                     |
|                     uint8_t_ b_PWM             : Selected PWM (0 or 1)        |
|                     uint8_t_ b_StartLevel      : Start period level selection |
|                                                0 : The period start with a |
|                                                    low level               |
|                                                1 : The period start with a |
|                                                    high level              |
|                     uint8_t_ b_StopMode        : Stop mode selection          |
|                                                0 : The PWM is stopped      |
|                                                    directly after the      |
|                                                    "i_APCI1710_DisablePWM" |
|                                                    function and break the  |
|                                                    last period             |
|                                                1 : After the               |
|                                                    "i_APCI1710_DisablePWM" |
|                                                     function the PWM is    |
|                                                     stopped at the end from|
|                                                     last period cycle.     |
|                     uint8_t_ b_StopLevel       : Stop PWM level selection     |
|                                                0 : The output signal keep  |
|                                                    the level after the     |
|                                                    "i_APCI1710_DisablePWM" |
|                                                    function                |
|                                                1 : The output signal is set|
|                                                    to low after the        |
|                                                    "i_APCI1710_DisablePWM" |
|                                                    function                |
|                                                2 : The output signal is set|
|                                                    to high after the       |
|                                                    "i_APCI1710_DisablePWM" |
|                                                    function                |
|                     uint8_t_ b_ExternGate      : Extern gate action selection |
|                                                0 : Extern gate signal not  |
|                                                    used.                   |
|                                                1 : Extern gate signal used.|
|                     uint8_t_ b_InterruptEnable : Enable or disable the PWM    |
|                                               interrupt.                   |
|                                               - APCI1710_ENABLE :          |
|                                                 Enable the PWM interrupt   |
|                                                 A interrupt occur after    |
|                                                 each period                |
|                                               - APCI1710_DISABLE :         |
|                                                 Disable the PWM interrupt  |
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value      : 0:  No error                                           |
|                    1:  Module selection wrong                             |
|                    2:  The module is not a PWM module                     |
|                    3:  PWM selection is wrong                             |
|                    4:  PWM not initialised see function                   |
|                         "i_APCI1710_InitPWM"                               |
|                    5:  PWM start level selection is wrong                 |
|                    6:  PWM stop mode selection is wrong                   |
|                    7:  PWM stop level selection is wrong                  |
|                    8:  Extern gate signal selection is wrong              |
|                    9: Interrupt parameter is wrong                       |
|                    10: Interrupt function not initialised.                |
|                         See function "i_APCI1710_SetBoardIntRoutineX"      |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_EnablePWM    (struct pci_dev *pdev,
		uint8_t  b_ModulNbr,
		uint8_t  b_PWM,
		uint8_t  b_StartLevel,
		uint8_t  b_StopMode,
		uint8_t  b_StopLevel,
		uint8_t  b_ExternGate,
		uint8_t  b_InterruptEnable)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_Status;
	uint32_t dw_Command;



	/**************************/
	/* Test the module number */
	/**************************/

	if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
	   {
	   /***************/
	   /* Test if PWM */
	   /***************/

		if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
	      /**************************/
	      /* Test the PWM selection */
	      /**************************/

	      if (b_PWM >= 0 && b_PWM <= 1)
		 {
		 /***************************/
		 /* Test if PWM initialised */
		 /***************************/

		 INPDW (GET_BAR2(pdev),
			(12 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
			 &dw_Status);

		 if (dw_Status & 0x10)
		    {
		    /**********************************/
		    /* Test the start level selection */
		    /**********************************/

		    if (b_StartLevel >= 0 && b_StartLevel <= 1)
		       {
		       /**********************/
		       /* Test the stop mode */
		       /**********************/

		       if (b_StopMode >= 0 && b_StopMode <= 1)
			  {
			  /***********************/
			  /* Test the stop level */
			  /***********************/

			  if (b_StopLevel >= 0 && b_StopLevel <= 2)
			     {
			     /*****************************/
			     /* Test the extern gate mode */
			     /*****************************/

			     if (b_ExternGate >= 0 && b_ExternGate <= 1)
				{
				/*****************************/
				/* Test the interrupt action */
				/*****************************/

				if (b_InterruptEnable == APCI1710_ENABLE || b_InterruptEnable == APCI1710_DISABLE)
				   {
				   /******************************************/
				   /* Test if interrupt function initialised */
				   /******************************************/

					if (((APCI1710_PRIVDATA(pdev)->s_InterruptInfos.b_InterruptInitialized) != 0)||(b_InterruptEnable == APCI1710_DISABLE))
							 {
				      /********************/
				      /* Read the command */
				      /********************/


					     INPDW (GET_BAR2(pdev),
					    		 8 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
								 &dw_Command);

				      dw_Command = dw_Command & 0x80;

				      /********************/
				      /* Make the command */
				      /********************/

				      dw_Command = dw_Command | b_StopMode | (b_InterruptEnable << 3) | (b_ExternGate << 4) | (b_StartLevel << 5);

				      if (b_StopLevel & 3)
					 {
					 dw_Command = dw_Command | 2;

					 if (b_StopLevel & 2)
					    {
					    dw_Command = dw_Command | 4;
					    }
					 }

				      APCI1710_PRIVDATA(pdev)->
				      s_ModuleInfo [(int)b_ModulNbr].
				      s_PWMModuleInfo.
				      s_PWMInfo [b_PWM].
				      b_InterruptEnable = b_InterruptEnable;

				      /*******************/
				      /* Set the command */
				      /*******************/

		    		     OUTPDW (GET_BAR2(pdev),
					    		 8 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
								 dw_Command);

				      /******************/
				      /* Enable the PWM */
				      /******************/

		    		     OUTPDW (GET_BAR2(pdev),
					    		 12 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
								 1);
				      }
				   else
				      {
				      /**************************************/
				      /* Interrupt function not initialised */
				      /**************************************/

				      i_ReturnValue = 10;
				      }
				   } // if (b_InterruptEnable == APCI1710_ENABLE || b_InterruptEnable == APCI1710_DISABLE)
				else
				   {
				   /********************************/
				   /* Interrupt parameter is wrong */
				   /********************************/

				   i_ReturnValue = 9;
				   } // if (b_InterruptEnable == APCI1710_ENABLE || b_InterruptEnable == APCI1710_DISABLE)
				} // if (b_ExternGate >= 0 && b_ExternGate <= 1)
			     else
				{
				/*****************************************/
				/* Extern gate signal selection is wrong */
				/*****************************************/

				i_ReturnValue = 8;
				} // if (b_ExternGate >= 0 && b_ExternGate <= 1)
			     } // if (b_StopLevel >= 0 && b_StopLevel <= 2)
			  else
			     {
			     /*************************************/
			     /* PWM stop level selection is wrong */
			     /*************************************/

			     i_ReturnValue = 7;
			     } // if (b_StopLevel >= 0 && b_StopLevel <= 2)
			  } // if (b_StopMode >= 0 && b_StopMode <= 1)
		       else
			  {
			  /************************************/
			  /* PWM stop mode selection is wrong */
			  /************************************/

			  i_ReturnValue = 6;
			  } // if (b_StopMode >= 0 && b_StopMode <= 1)
		       } // if (b_StartLevel >= 0 && b_StartLevel <= 1)
		    else
		       {
		       /**************************************/
		       /* PWM start level selection is wrong */
		       /**************************************/

		       i_ReturnValue = 5;
		       } // if (b_StartLevel >= 0 && b_StartLevel <= 1)
		    } // if (dw_Status & 0x10)
		 else
		    {
		    /***********************/
		    /* PWM not initialised */
		    /***********************/

		    i_ReturnValue = 4;
		    } // if (dw_Status & 0x10)
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int i_APCI1710_DisablePWM (uint8_t_  b_BoardHandle,     |
|                                                  uint8_t_  b_ModulNbr,        |
|                                                  uint8_t_  b_PWM)             |
+----------------------------------------------------------------------------+
| Task              : Disable the selected PWM (b_PWM) from selected module  |
|                     (b_ModulNbr). The output signal level depend of the    |
|                     initialisation by the "i_APCI1710_EnablePWM".          |
|                     See the b_StartLevel, b_StopMode and b_StopLevel       |
|                     parameters from this function.                         |
+----------------------------------------------------------------------------+
| Input Parameters  :uint8_t_ b_BoardHandle : Handle of board APCI-1710         |
|                    uint8_t_ b_ModulNbr    : Selected module number (0 to 3)   |
|                    uint8_t_ b_PWM         : Selected PWM (0 or 1)             |
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value      :  0: No error                                           |
|                     1: Module selection wrong                             |
|                     2: The module is not a PWM module                     |
|                     3: PWM selection is wrong                             |
|                     4: PWM not initialised see function                   |
|                         "i_APCI1710_InitPWM"                               |
|                     5: PWM not enabled see function                       |
|                         "i_APCI1710_EnablePWM"                             |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_DisablePWM   (struct pci_dev *pdev,
		uint8_t  b_ModulNbr,
		uint8_t  b_PWM)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_Status;

	/**************************/
	/* Test the module number */
	/**************************/

	if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
	   {
	   /***************/
	   /* Test if PWM */
	   /***************/

		if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
	      /**************************/
	      /* Test the PWM selection */
	      /**************************/

	      if (b_PWM >= 0 && b_PWM <= 1)
		 {
		 /***************************/
		 /* Test if PWM initialised */
		 /***************************/

		 INPDW (GET_BAR2(pdev),
			(12 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
			 &dw_Status);

		 if (dw_Status & 0x10)
		    {
		    /***********************/
		    /* Test if PWM enabled */
		    /***********************/

		    if (dw_Status & 0x1)
		       {
		       /*******************/
		       /* Disable the PWM */
		       /*******************/

  		     OUTPDW (GET_BAR2(pdev),
			    		 12 + (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr),
						 0);
		       } // if (dw_Status & 0x1)
		    else
		       {
		       /*******************/
		       /* PWM not enabled */
		       /*******************/

		       i_ReturnValue = 5;
		       } // if (dw_Status & 0x1)
		    } // if (dw_Status & 0x10)
		 else
		    {
		    /***********************/
		    /* PWM not initialised */
		    /***********************/

		    i_ReturnValue = 4;
		    } // if (dw_Status & 0x10)
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int i_APCI1710_GetPWMInitialisation                  |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr,               |
|                                        uint8_t_     b_PWM,                    |
|                                        Puint8_t_   pb_TimingUnit,             |
|                                        Puint32_t pul_LowTiming,              |
|                                        Puint32_t pul_HighTiming,             |
|                                        Puint8_t_   pb_StartLevel,             |
|                                        Puint8_t_   pb_StopMode,               |
|                                        Puint8_t_   pb_StopLevel,              |
|                                        Puint8_t_   pb_ExternGate,             |
|                                        Puint8_t_   pb_InterruptEnable,        |
|                                        Puint8_t_   pb_Enable)                 |
+----------------------------------------------------------------------------+
| Task              : Return the PWM (b_PWM) initialisation from selected    |
|                     module (b_ModulNbr). You must calling the              |
|                     "i_APCI1710_InitPWM" function be for you call this     |
|                     function.                                              |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle : Handle of board APCI-1710        |
|                     uint8_t_ b_ModulNbr    : Selected module number (0 to 3)  |
|                     uint8_t_ b_PWM         : Selected PWM (0 or 1)            |
+----------------------------------------------------------------------------+
| Output Parameters : Puint8_t_  pb_TimingUnit      : Base timing Unit (0 to 4) |
|                                                       0 : ns               |
|                                                       1 : �s               |
|                                                       2 : ms               |
|                                                       3 : s                |
|                                                       4 : mn               |
|                     Puint32_t pul_LowTiming      : Low base timing value.    |
|                     Puint32_t pul_HighTiming     : High base timing value.   |
|                     Puint8_t_  pb_StartLevel      : Start period level        |
|                                                  selection                 |
|                                                       0 : The period start |
|                                                           with a low level |
|                                                       1 : The period start |
|                                                           with a high level|
|                     Puint8_t_  pb_StopMode        : Stop mode selection       |
|                                                  0 : The PWM is stopped    |
|                                                      directly after the    |
|                                                     "i_APCI1710_DisablePWM"|
|                                                      function and break the|
|                                                      last period           |
|                                                  1 : After the             |
|                                                     "i_APCI1710_DisablePWM"|
|                                                      function the PWM is   |
|                                                      stopped at the end    |
|                                                      from last period cycle|
|                     Puint8_t_  pb_StopLevel        : Stop PWM level selection |
|                                                    0 : The output signal   |
|                                                        keep the level after|
|                                                        the                 |
|                                                     "i_APCI1710_DisablePWM"|
|                                                        function            |
|                                                    1 : The output signal is|
|                                                        set to low after the|
|                                                     "i_APCI1710_DisablePWM"|
|                                                        function            |
|                                                    2 : The output signal is|
|                                                        set to high after   |
|                                                        the                 |
|                                                     "i_APCI1710_DisablePWM"|
|                                                        function            |
|                     Puint8_t_  pb_ExternGate      : Extern gate action        |
|                                                  selection                 |
|                                                   0 : Extern gate signal   |
|                                                       not used.            |
|                                                   1 : Extern gate signal   |
|                                                       used.                |
|                     Puint8_t_  pb_InterruptEnable : Enable or disable the PWM |
|                                                  interrupt.                |
|                                                  - APCI1710_ENABLE :       |
|                                                    Enable the PWM interrupt|
|                                                    A interrupt occur after |
|                                                    each period             |
|                                                  - APCI1710_DISABLE :      |
|                                                    Disable the PWM         |
|                                                    interrupt               |
|                     Puint8_t_  pb_Enable          : Indicate if the PWM is    |
|                                                  enabled or no             |
|                                                       0 : PWM not enabled  |
|                                                       1 : PWM enabled      |
+----------------------------------------------------------------------------+
| Return Value      :  0: No error                                           |
|                     1: Module selection wrong                             |
|                     2: The module is not a PWM module                     |
|                     3: PWM selection is wrong                             |
|                     4: PWM not initialised see function                   |
|                         "i_APCI1710_InitPWM"                               |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_GetPWMInitialisation (struct pci_dev *pdev,
		 uint8_t     b_ModulNbr,
		 uint8_t     b_PWM,
		 uint8_t   *pb_TimingUnit,
		 uint32_t *pul_LowTiming,
		 uint32_t *pul_HighTiming,
		 uint8_t   *pb_StartLevel,
		 uint8_t   *pb_StopMode,
		 uint8_t   *pb_StopLevel,
		 uint8_t   *pb_ExternGate,
		 uint8_t   *pb_InterruptEnable,
		 uint8_t   *pb_Enable)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_Status;
	uint32_t dw_Command;

	/**************************/
	/* Test the module number */
	/**************************/

	if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
	   {
	   /***************/
	   /* Test if PWM */
	   /***************/

		if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
	      /**************************/
	      /* Test the PWM selection */
	      /**************************/

	      if (b_PWM >= 0 && b_PWM <= 1)
		 {
		 /***************************/
		 /* Test if PWM initialised */
		 /***************************/

		 INPDW (GET_BAR2(pdev),
			(12 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
			 &dw_Status);

		 if (dw_Status & 0x10)
		    {
		    /***********************/
		    /* Read the low timing */
		    /***********************/

			 INPDW (GET_BAR2(pdev),
				(0 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
				pul_LowTiming);

		    /************************/
		    /* Read the high timing */
		    /************************/

			 INPDW (GET_BAR2(pdev),
				(4 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
				pul_HighTiming);

		    /********************/
		    /* Read the command */
		    /********************/

			 INPDW (GET_BAR2(pdev),
				(8 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
				 &dw_Command);

		    *pb_StartLevel      = (uint8_t) ((dw_Command >> 5) & 1);
		    *pb_StopMode        = (uint8_t) ((dw_Command >> 0) & 1);
		    *pb_StopLevel       = (uint8_t) ((dw_Command >> 1) & 1);
		    *pb_ExternGate      = (uint8_t) ((dw_Command >> 4) & 1);
		    *pb_InterruptEnable = (uint8_t) ((dw_Command >> 3) & 1);

		    if (*pb_StopLevel)
		       {
		       *pb_StopLevel = *pb_StopLevel + (uint8_t) ((dw_Command >> 2) & 1);
		       }

		    /********************/
		    /* Read the command */
		    /********************/

			 INPDW (GET_BAR2(pdev),
				(8 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
				 &dw_Command);

		    *pb_Enable = (uint8_t) ((dw_Command >> 0) & 1);

		    *pb_TimingUnit = APCI1710_PRIVDATA(pdev)->
					s_ModuleInfo [(int)b_ModulNbr].
				     s_PWMModuleInfo.
				     s_PWMInfo [b_PWM].
				     b_TimingUnit;
		    } // if (dw_Status & 0x10)
		 else
		    {
		    /***********************/
		    /* PWM not initialised */
		    /***********************/

		    i_ReturnValue = 4;
		    } // if (dw_Status & 0x10)
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int i_APCI1710_GetPWMStatus                          |
|                               (uint8_t_    b_BoardHandle,                     |
|                                uint8_t_    b_ModulNbr,                        |
|                                uint8_t_    b_PWM,                             |
|                                Puint8_t_  pb_PWMOutputStatus,                 |
|                                Puint8_t_  pb_ExternGateStatus)                |
+----------------------------------------------------------------------------+
| Task              : Return the status from selected PWM (b_PWM) from       |
|                     selected module (b_ModulNbr).                          |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_  b_BoardHandle : Handle of board APCI-1710       |
|                     uint8_t_  b_PWM         : Selected PWM (0 or 1)           |
|                     uint8_t_  b_ModulNbr    : Selected module number (0 to 3) |
+----------------------------------------------------------------------------+
| Output Parameters : Puint8_t_  pb_PWMOutputStatus  : Return the PWM output    |
|                                                   level status.            |
|                                                    0 : The PWM output level|
|                                                        is low.             |
|                                                    1 : The PWM output level|
|                                                        is high.            |
|                     Puint8_t_  pb_ExternGateStatus : Return the extern gate   |
|                                                   level status.            |
|                                                    0 : The extern gate is  |
|                                                        low.                |
|                                                    1 : The extern gate is  |
|                                                        high.               |
+----------------------------------------------------------------------------+
| Return Value      :  0: No error                                           |
|                     1: Module selection wrong                             |
|                     2: The module is not a PWM module                     |
|                     3: PWM selection is wrong                             |
|                     4: PWM not initialised see function                   |
|                         "i_APCI1710_InitPWM"                               |
|                     5: PWM not enabled see function "i_APCI1710_EnablePWM"|
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_GetPWMStatus (struct pci_dev *pdev,
		uint8_t    b_ModulNbr,
		uint8_t    b_PWM,
		uint8_t  *pb_PWMOutputStatus,
		uint8_t  *pb_ExternGateStatus)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_Status;


	/**************************/
		/* Test the module number */
		/**************************/

		if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
		   {
		   /***************/
		   /* Test if PWM */
		   /***************/

			if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
		      {
		      /**************************/
		      /* Test the PWM selection */
		      /**************************/

		      if (b_PWM >= 0 && b_PWM <= 1)
			 {
			 /***************************/
			 /* Test if PWM initialised */
			 /***************************/

			 INPDW (GET_BAR2(pdev),
				(12 +  (20 * b_PWM) + MODULE_OFFSET(b_ModulNbr)),
				 &dw_Status);

		 if (dw_Status & 0x10)
		    {
		    /***********************/
		    /* Test if PWM enabled */
		    /***********************/

		    if (dw_Status & 0x1)
		       {
		       *pb_PWMOutputStatus  = (uint8_t) ((dw_Status >> 7) & 1);
		       *pb_ExternGateStatus = (uint8_t) ((dw_Status >> 6) & 1);
		       } // if (dw_Status & 0x1)
		    else
		       {
		       /*******************/
		       /* PWM not enabled */
		       /*******************/

		       i_ReturnValue = 5;
		       } // if (dw_Status & 0x1)
		    } // if (dw_Status & 0x10)
		 else
		    {
		    /***********************/
		    /* PWM not initialised */
		    /***********************/

		    i_ReturnValue = 4;
		    } // if (dw_Status & 0x10)
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      else
		 {
		 /******************************/
		 /* Tor PWM selection is wrong */
		 /******************************/

		 i_ReturnValue = 3;
		 } // if (b_PWM >= 0 && b_PWM <= 1)
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int   i_APCI1710_ReadPWM1DigitalInput                |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr,               |
|                                        uint8_t_     b_InputChannel,           |
|                                        Puint8_t_   pb_ChannelStatus)          |
+----------------------------------------------------------------------------+
| Task              : Read the status from selected PWM digital input        |
|                     (b_InputChannel)                                       |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle         : Handle of board APCI-1710|
|                     uint8_t_ b_ModulNbr            : Module number to         |
|                                                   configure (0 to 3)       |
|                     uint8_t_ b_InputChannel        : Selection from digital   |
|                                                   input ( 0 to 2)          |
+----------------------------------------------------------------------------+
| Output Parameters : Puint8_t_  pb_ChannelStatus    : Digital input channel    |
|                                                   status                   |
|                                                   0 : Channle is not active|
|                                                   1 : Channle is active    |
+----------------------------------------------------------------------------+
| Return Value      : 0: No error                                            |
|                    1: The module parameter is wrong                       |
|                    2: The module is not a PWM module                      |
|                    3: Firmware revision error                             |
|                    4: The selected PWM digital input is wrong             |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_ReadPWM1DigitalInput  (struct pci_dev *pdev,
		uint8_t    b_ModulNbr,
		uint8_t    b_InputChannel,
		uint8_t  *pb_ChannelStatus)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_StatusReg;

	/**************************/
	/* Test the module number */
	/**************************/

		if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
		  {
	   /***********************/
	   /* Test if PWM counter */
	   /***********************/

	if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
	      {
              /*****************************/
	      /* Test the Firmware version */
	      /*****************************/

	      if ((APCI1710_PRIVDATA(pdev)->
		   s_BoardInfos.
		   dw_MolduleConfiguration [b_ModulNbr] & 0xFFFF) >= 0x3132)
	         {
	         /******************************************/
	         /* Test the digital imnput channel number */
	         /******************************************/

	         if ((b_InputChannel >= 0) && (b_InputChannel <= 2))
		    {
		    /**************************/
		    /* Read all digital input */
		    /**************************/

			 INPDW (GET_BAR2(pdev),
				(44 + MODULE_OFFSET(b_ModulNbr)),
				 &dw_StatusReg);

		    *pb_ChannelStatus = (uint8_t) ((dw_StatusReg >> b_InputChannel) & 1);
		    }
	         else
		    {
		    /********************************/
		    /* Selected digital input error */
		    /********************************/

		    i_ReturnValue = 4;
		    }
                 }
	      else
	         {
		 /***************************/
		 /* Firmware revision error */
		 /***************************/
		 
		 i_ReturnValue = 3;
		 }
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+--------------------------------------------uint8_t--------------------------------+
| Function Name     : int   i_APCI1710_ReadPWMAllDigitalInput              |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr,               |
|                                        Puint8_t_   pb_InputStatus)            |
+----------------------------------------------------------------------------+
| Task              : Read the status from all PWM digital inputs from       |
|                     selected PWM module (b_ModulNbr)                       |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle         : Handle of board APCI-1710|
|                     uint8_t_ b_ModulNbr            : Module number to         |
|                                                   configure (0 to 3)       |
+----------------------------------------------------------------------------+
| Output Parameters : Puint8_t_  pb_InputStatus      : Digital inputs channel   |
|                                                   status                   |
+----------------------------------------------------------------------------+
| Return Value      : 0: No error                                            |
|                    1: The module parameter is wrong                       |
|                    2: The module is not a PWM module                      |
|                    3: Firmware revision error                             |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_ReadPWMAllDigitalInput  (struct pci_dev *pdev,
		uint8_t     b_ModulNbr,
		uint8_t   *pb_InputStatus)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_StatusReg;



	/**************************/
		/* Test the module number */
		/**************************/

			if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
			  {
		   /***********************/
		   /* Test if PWM counter */
		   /***********************/

		  if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
		      {
	           /*****************************/
		      /* Test the Firmware version */
		      /*****************************/

		      if ((APCI1710_PRIVDATA(pdev)->
			   s_BoardInfos.
			   dw_MolduleConfiguration [b_ModulNbr] & 0xFFFF) >= 0x3132)
		         {
	         /**************************/
	         /* Read all digital input */
	         /**************************/

			 INPDW (GET_BAR2(pdev),
				(44 + MODULE_OFFSET(b_ModulNbr)),
				 &dw_StatusReg);


	         *pb_InputStatus = (uint8_t) dw_StatusReg;
		 }
	      else
	         {
		 /***************************/
		 /* Firmware revision error */
		 /***************************/
		 
		 i_ReturnValue = 3;
		 }
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}


/*
+----------------------------------------------------------------------------+
| Function Name     : int                                                  |
|                      i_APCI1710_EnableDisablePWMDigitalOutputManualSetting |
|                                      (uint8_t_     b_BoardHandle,             |
|                                       uint8_t_     b_ModulNbr,                |
|                                       uint8_t_     b_Flag)                    |
+----------------------------------------------------------------------------+
| Task              : Enable or disable the manually settings from output H  |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle : Handle of board APCI-1710        |
|                     uint8_t_ b_ModulNbr    : Module number to configure       |
|                     uint8_t_ b_Flag        : APCI1710_DISABLE : Disable the   |
|                                              manually settings.            |
|					       This is the default state.    |
|					       Output H is parallel connected|
|					       to the output A (PWM0)        |
|				            APCI1710_ENABLE  : Enable the    |
|					       manually seting from output H |
|					       vi the functions              |
|					  - i_APCI1710_SetPWMDigitalOutputOn |
|					  - i_APCI1710_SetPWMDigitalOutputOff|
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value      : 0: No error                                            |
|                    1: The module is not a PWM module                      |
|                    2: Firmware revision error                             |
|                    3: Flag parameter is wrong                             |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_EnableDisablePWMDigitalOutputManualSetting  (struct pci_dev *pdev,
		uint8_t     b_ModulNbr,
		uint8_t     b_Flag)
	{
	int    i_ReturnValue = 0;
	uint32_t dw_Status      = 0;


	/**************************/
		/* Test the module number */
		/**************************/

			if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
			  {
		   /***********************/
		   /* Test if PWM  */
		   /***********************/

		  if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
		      {

		 /*****************/
		 /* Test the flag */
		 /*****************/

		 if ((b_Flag == APCI1710_ENABLE) || (b_Flag == APCI1710_DISABLE))
		    {
	            /*****************/
	            /* Save the flag */
	            /*****************/

		    APCI1710_PRIVDATA(pdev)->
		    s_ModuleInfo [(int)b_ModulNbr].
		    s_PWMModuleInfo.
		    b_DigitalOutputFlag = b_Flag;

		    /*************************/
		    /* Set the configuration */
		    /*************************/

				 INPDW (GET_BAR2(pdev),
					(40 + MODULE_OFFSET(b_ModulNbr)),
					 &dw_Status);

	            if (b_Flag)
		       {
	               dw_Status = dw_Status | 2;
		       }
		    else
		       {
	               dw_Status = dw_Status & 0xFFFFFFDFUL;
		       }

	            OUTPDW (GET_BAR2(pdev),
					(40 + MODULE_OFFSET(b_ModulNbr)),
					 dw_Status);
		    }
	         else
	            {
		    /**************/
		    /* Flag error */
		    /**************/
		 
		    i_ReturnValue = 3;
		    }

	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}


/*
+----------------------------------------------------------------------------+
| Function Name     : int   i_APCI1710_SetPWMDigitalOutputOn               |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr)               |
+----------------------------------------------------------------------------+
| Task              : Set the digital output from selected PWM moule         |
|                     (b_ModuleNbr) ON                                       |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle         : Handle of board APCI-1710|
|                     uint8_t_ b_ModulNbr            : Module number to         |
|                                                   configure (0 to 3)       |
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value      : 0: No error                                            |
|                    1: The module is not a PWM module                      |
|                    2: Firmware revision error                             |
|                    3: Manually settings option not active                 |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_SetPWMDigitalOutputOn        (struct pci_dev *pdev,
		uint8_t     b_ModulNbr)

	{
	uint8_t    i_ReturnValue = 0;




	/**************************/
		/* Test the module number */
		/**************************/

			if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
			  {
		   /***********************/
		   /* Test if PWM counter */
		   /***********************/

		  if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
		      {

		 /********************************************/
		 /* Test if manually setting option activate */
		 /********************************************/

		 if (APCI1710_PRIVDATA(pdev)->
			 s_ModuleInfo [(int)b_ModulNbr].
		     s_PWMModuleInfo.
		     b_DigitalOutputFlag == APCI1710_ENABLE)
		    {
	            /*****************************/
	            /* Set the digital output ON */
	            /*****************************/
	            OUTPDW (GET_BAR2(pdev),
					(40 + MODULE_OFFSET(b_ModulNbr)),
					 3);
		    }
		 else
		    {
		    /***************************************/
		    /* Manually settings option not active */
		    /***************************************/

		    i_ReturnValue = 3;
		    }
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}

/*
+----------------------------------------------------------------------------+
| Function Name     : int   i_APCI1710_SetPWMDigitalOutputOff              |
|                                       (uint8_t_     b_BoardHandle,            |
|                                        uint8_t_     b_ModulNbr)               |
+----------------------------------------------------------------------------+
| Task              : Set the digital output from selected PWM moule         |
|                     (b_ModuleNbr) OFF                                      |
+----------------------------------------------------------------------------+
| Input Parameters  : uint8_t_ b_BoardHandle         : Handle of board APCI-1710|
|                     uint8_t_ b_ModulNbr            : Module number to         |
|                                                   configure (0 to 3)       |
+----------------------------------------------------------------------------+
| Output Parameters : -                                                      |
+----------------------------------------------------------------------------+
| Return Value      : 0: No error                                            |
|                    1: The module is not a PWM module                      |
|                    2: Firmware revision error                             |
|                    3: Manually settings option not active                 |
+----------------------------------------------------------------------------+
*/

int   i_APCI1710_SetPWMDigitalOutputOff         (struct pci_dev *pdev,
		uint8_t     b_ModulNbr)

	{
	uint8_t    i_ReturnValue = 0;




	/**************************/
		/* Test the module number */
		/**************************/

			if (b_ModulNbr < APCI1710_PRIVDATA(pdev)->s_BoardInfos.b_NumberOfModule)
			  {
		   /***********************/
		   /* Test if PWM counter */
		   /***********************/

		   if (APCI1710_MODULE_FUNCTIONALITY(pdev,b_ModulNbr) == APCI1710_PWM)
		      {
		    	  /********************************************/
		    	  		 /* Test if manually setting option activate */
		    	  		 /********************************************/

		    	  		 if (APCI1710_PRIVDATA(pdev)->
		    	  			 s_ModuleInfo [(int)b_ModulNbr].
		    	  		     s_PWMModuleInfo.
		    	  		     b_DigitalOutputFlag == APCI1710_ENABLE)
		    	  		    {
		    	  	            /*****************************/
		    	  	            /* Set the digital output ON */
		    	  	            /*****************************/
		    	  	            OUTPDW (GET_BAR2(pdev),
		    	  					(40 + MODULE_OFFSET(b_ModulNbr)),
		    	  					 2);
		    	  		    }
		 else
		    {
		    /***************************************/
		    /* Manually settings option not active */
		    /***************************************/

		    i_ReturnValue = 3;
		    }
	      }
	   else
	      {
	      /**********************************/
	      /* The module is not a PWM module */
	      /**********************************/

	      i_ReturnValue = 2;
	      }
	   }
	else
	   {
	   /***********************/
	   /* Module number error */
	   /***********************/

	   i_ReturnValue = 1;
	   }

	return (i_ReturnValue);
	}



