/** @file sample_imp_cpt_apci1710.c
 
   This demonstrates how to use the impulse counter in user mode.
 
   @par CREATION  
   @author S. Weber
   @date   06.10.08
   
   @par VERSION
   @verbatim
   $LastChangedRevision:$
   $LastChangedDate:$
   @endverbatim   
   
   @par LICENCE
   @verbatim
    Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
    ADDI-DATA GmbH
    Airpark Business Center
    Airport Boulevard B210
    77836 Rheinmuenster
    Germany
    Tel: +49(0)7229/1847-0
    Fax: +49(0)7229/1847-200
    http://www.addi-data-com
    info@addi-data.com
        
   This program is free software; you can redistribute it and/or modify it under 
   the terms of the GNU General Public License as published by the Free Software 
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. 
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with 
   this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   You shoud also find the complete GPL in the COPYING file 
   accompanying this source code.
   @endverbatim   
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <string.h>

#include <apci1710.h>

#include "sample_apci1710_common.h"

//------------------------------------------------------------------------------

static int gi_fd; /* Global file handle on the board to use */

//--------------------------------------------------------------------------------
/**
 * @brief Sample function for impulse counter usage.
 * @param fd File descriptor of the opened APCI-1710 device
 */
void sample (int fd)
{
    uint8_t   b_ModuleNbr       = 0;
    uint8_t   b_PulseEncoderNbr = 0;
    uint8_t   i = 0;
    uint8_t   b_ArgArray[4];    
    uint32_t ui_ArgArray[5];    
    int       i_ReturnValue = 0;
        
    gi_fd = fd;

    /* Initialise the impulse counter */
    ui_ArgArray[0] = (uint32_t) b_ModuleNbr;    // The module to use (0 to 3).
    ui_ArgArray[1] = (uint32_t) b_PulseEncoderNbr;    // Pulse encoder selection (0 to 3)
    ui_ArgArray[2] = 1;                // Input level selection (0 or 1)
    ui_ArgArray[3] = 0;                // Digital TRIGGER output action
    ui_ArgArray[4] = 1000;                // Pulse encoder start,reload value
                            
    i_ReturnValue = ioctl (fd,CMD_APCI1710_InitPulseEncoder, ui_ArgArray);

    /* Test the return value */
    switch (i_ReturnValue)
        {
        case 0:
        printf ("\nCMD_APCI1710_InitPulseEncoder OK");

        /* Enable the impulse counter */
        b_ArgArray[0] = b_ModuleNbr;        // The module to use (0 to 3).
        b_ArgArray[1] = b_PulseEncoderNbr;    // Pulse encoder selection (0 to 3)
        b_ArgArray[2] = APCI1710_CONTINUOUS;    // Cycle selection (APCI1710_CONTINUOUS OR APCI1710_SINGLE)
        b_ArgArray[3] = 0;            // Interrupt handling
                            
        i_ReturnValue = ioctl (fd,CMD_APCI1710_EnablePulseEncoder, b_ArgArray);

        /* Test the return value */
        switch (i_ReturnValue)
            {
            case 0:
            printf ("\nCMD_APCI1710_EnablePulseEncoder OK");

            printf ("\n\n");

            do
                {
                /* Read the impulse counter status */
                b_ArgArray[0] = b_ModuleNbr;        // The module to use (0 to 3).
                b_ArgArray[1] = b_PulseEncoderNbr;    // Pulse encoder selection (0 to 3)
                    
                i_ReturnValue = ioctl (fd,CMD_APCI1710_ReadPulseEncoderStatus, b_ArgArray);

                /* Test the return value */
                switch (i_ReturnValue)
                    {
                    case 0:
                    /* Read the impulse counter status */
                    ui_ArgArray[0] = (uint32_t) b_ModuleNbr;    // The module to use (0 to 3).
                    ui_ArgArray[1] = (uint32_t) b_PulseEncoderNbr;    // Pulse encoder selection (0 to 3)
                    
                    i_ReturnValue = ioctl (fd,CMD_APCI1710_ReadPulseEncoderValue, ui_ArgArray);

                    /* Test the return value */
                    switch (i_ReturnValue)
                        {
                        case 0:
                        printf ("\nImpuls counter value from module %d counter %d = %u. Overflow = %d", b_ModuleNbr, b_PulseEncoderNbr, ui_ArgArray[0], b_ArgArray[0]);
                        break;

                        case 1:
                        printf ("\nCMD_APCI1710_ReadPulseEncoderValue error");
                        printf ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                        break;

                        case 2:
                        printf ("\nCMD_APCI1710_ReadPulseEncoderValue error");
                        printf ("\nError = %d. Module selection is wrong", i_ReturnValue);
                        break;

                        case 3:
                        printf ("\nCMD_APCI1710_ReadPulseEncoderValue error");
                        printf ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
                        break;

                        case 4:
                        printf ("\nCMD_APCI1710_ReadPulseEncoderValue error");
                        printf ("\nError = %dPulse encoder not initialised", i_ReturnValue);
                        break;

                        default:
                        printf ("\nCMD_APCI1710_ReadPulseEncoderValue error");
                        printf ("\nError = %d.", i_ReturnValue);
                        perror("\nioctl");                            
                        break;
                        } // switch (i_ReturnValue)
                    break;

                    case 1:
                    printf ("\nCMD_APCI1710_ReadPulseEncoderStatus error");
                    printf ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                    break;

                    case 2:
                    printf ("\nCMD_APCI1710_ReadPulseEncoderStatus error");
                    printf ("\nError = %d. Module selection is wrong", i_ReturnValue);
                    break;

                    case 3:
                    printf ("\nCMD_APCI1710_ReadPulseEncoderStatus error");
                    printf ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
                    break;

                    case 4:
                    printf ("\nCMD_APCI1710_ReadPulseEncoderStatus error");
                    printf ("\nError = %dPulse encoder not initialised", i_ReturnValue);
                    break;

                    default:
                    printf ("\nCMD_APCI1710_ReadPulseEncoderStatus error");
                    printf ("\nError = %d.", i_ReturnValue);
                    perror("\nioctl");                            
                    break;
                    } // switch (i_ReturnValue)

                sleep (1);
                }
            while (i++ < 100);                       

            /* Disable the impulse counter */
            b_ArgArray[0] = b_ModuleNbr;        // The module to use (0 to 3).
            b_ArgArray[1] = b_PulseEncoderNbr;    // Pulse encoder selection (0 to 3)
                            
            i_ReturnValue = ioctl (fd,CMD_APCI1710_DisablePulseEncoder, b_ArgArray);

            /* Test the return value */
            switch (i_ReturnValue)
                {
                case 0:
                printf ("\nCMD_APCI1710_DisablePulseEncoder OK");
                break;

                case 1:
                printf ("\nCMD_APCI1710_DisablePulseEncoder error");
                printf ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
                break;

                case 2:
                printf ("\nCMD_APCI1710_DisablePulseEncoder error");
                printf ("\nError = %d. Module selection is wrong", i_ReturnValue);
                break;

                case 3:
                printf ("\nCMD_APCI1710_DisablePulseEncoder error");
                printf ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
                break;

                case 4:
                printf ("\nCMD_APCI1710_DisablePulseEncoder error");
                printf ("\nError = %dPulse encoder not initialised", i_ReturnValue);
                break;

                default:
                printf ("\nCMD_APCI1710_DisablePulseEncoder error");
                printf ("\nError = %d.", i_ReturnValue);
                perror("\nioctl");                            
                break;
                } // switch (i_ReturnValue)
            break;

            case 1:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
            break;

            case 2:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Module selection is wrong", i_ReturnValue);
            break;

            case 3:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
            break;

            case 4:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Pulse encoder not initialised", i_ReturnValue);
            break;

            case 5:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Cycle selection mode is wrong", i_ReturnValue);
            break;

            case 6:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Interrupt handling mode is wrong", i_ReturnValue);
            break;

            case 7:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d. Interrupt routine not installed", i_ReturnValue);
            break;

            default:
            printf ("\nCMD_APCI1710_EnablePulseEncoder error");
            printf ("\nError = %d.", i_ReturnValue);
            perror("\nioctl");                    
            break;
            }



        break;

        case 1:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. The handle parameter of the board is wrong", i_ReturnValue);
        break;

        case 2:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. The module is not a pulse encoder module", i_ReturnValue);
        break;

        case 3:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. Pulse encoder selection is wrong", i_ReturnValue);
        break;

        case 4:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. Input level selection is wrong", i_ReturnValue);
        break;

        case 5:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. Digital TRIGGER output action selection is wrong", i_ReturnValue);
        break;

        case 6:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d. Pulse encoder start value is wrong", i_ReturnValue);
        break;

        default:
        printf ("\nCMD_APCI1710_InitPulseEncoder error");
        printf ("\nError = %d.", i_ReturnValue);
        perror("\nioctl");                    
        break;
        } // switch (i_ReturnValue)


    printf ("\nProgram concluded.");
}

//--------------------------------------------------------------------------------

int main(int argc, char** argv)
{

    int * apci1710_card_fd;
    unsigned int apci1710_card_number = 0;
    {
        apci1710_card_number = apci1710_find_cards(&apci1710_card_fd);
        
        printf("total: %d cards\n",apci1710_card_number);
    }    
    
    {
        int i;
        for (i=0;i< apci1710_card_number; i++)
        {
            printf("\n+++++++++++++++++++++++++++++++++++++\nrunning test on device of minor number %d\n+++++++++++++++++++++++++++++++++++++\n\n\n",i);
            
            /* Compare logic sample */
            sample (apci1710_card_fd[i]);    
        }
    }

    

    return 0;
}
//--------------------------------------------------------------------------------
