/** @file sample_apci1710_moduleids.c
 
   Demonstrate how to get module ID.
 
   @par CREATION  
   @author Sylvain Nahas
   @date   21/11/2007
   
   @par VERSION
   @verbatim
   $LastChangedRevision:$
   $LastChangedDate:$
   @endverbatim   
   
   @par LICENCE
   @verbatim
    Copyright (C) 2009  ADDI-DATA GmbH for the source code of this module.
        
    ADDI-DATA GmbH
    Airpark Business Center
    Airport Boulevard B210
    77836 Rheinm�nster
    Germany
    Tel: +49(0)7229/1847-0
    Fax: +49(0)7229/1847-200
    http://www.addi-data-com
    info@addi-data.com
        
   This program is free software; you can redistribute it and/or modify it under 
   the terms of the GNU General Public License as published by the Free Software 
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. 
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with 
   this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   You shoud also find the complete GPL in the COPYING file 
   accompanying this source code.
   @endverbatim   
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <apci1710.h>

#include "sample_apci1710_common.h"

//--------------------------------------------------------------------------------
// Output module functionality (configuration/ID register)  for all 4 modules

/**
 * @brief Convert functionality ID to string
 * @param Functionality Functionality ID
 * @return String representation of the functionality ID
 */
static const char * functionality_id_to_str(uint32_t Functionality)
{
    switch(Functionality)
    {
        case APCI1710_INCREMENTAL_COUNTER:
            return "INCREMENTAL_COUNTER";
        case APCI1710_SSI_COUNTER:
            return "SSI_COUNTER";
        case APCI1710_TTL_IO:
            return "TTL_IO";
        case APCI1710_DIGITAL_IO:
            return "DIGITAL_IO";
        case APCI1710_82X54_TIMER:
            return "82X54_TIMER";
        case APCI1710_CHRONOMETER:
            return "CHRONOMETER";
        case APCI1710_PULSE_ENCODER:
            return "PULSE_ENCODER";
        case APCI1710_TOR_COUNTER:
            return "TOR_COUNTER";
        case APCI1710_PWM:
            return "PWM";
        case APCI1710_ETM:
            return "ETM";
        case APCI1710_CDA:
            return "CDA";
	case APCI1710_BISS_MASTER:
	    return "BISS";
        default:
            return "UNKNOWN";
    }
} 
//--------------------------------------------------------------------------------
/**
 * @brief Print module IDs for a given board
 * @param fd File descriptor of the opened APCI-1710 device
 */
int print_modules_id(int fd, int board)
{
    uint32_t IDs[4];
    
    if (ioctl (fd,CMD_APCI1710_GetModulesId,IDs))
    {
        perror("CMD_APCI1710_GetModulesId");
        return -1;
    }
    
    /* output functionality */
    {
        int i;
        for (i=0;i<4; i++)
        {
            uint32_t Functionality = (IDs[i]>>16) & 0x0000FFFF;
            uint32_t Version = (IDs[i]) & 0x0000FFFF;
            printf("board %d : module %d has ID 0x%.8X (functionality: %s (0x%.4X) version: 0x%.4X)\n",board,i,IDs[i],functionality_id_to_str(Functionality),Functionality,Version);
        }
    }
    return 0;    
}


//--------------------------------------------------------------------------------


int main(int argc, char** argv)
{

    int * apci1710_card_fd;
    unsigned int apci1710_card_number = 0;
    {
        apci1710_card_number = apci1710_find_cards(&apci1710_card_fd);    
        printf("total: %d boards\n",apci1710_card_number);
    }    
    
    {
        int i;
        for (i=0;i< apci1710_card_number; i++)
        {
            print_modules_id(apci1710_card_fd[i],i);    
        }
    }

    

    return 0;
}
//--------------------------------------------------------------------------------
