/** @file PCI3300.h
 *
 *  @authors S. Beuque
 *  @date    Creation: 06.10.2014
 *
 *  @par DESCRIPTION
 *  Main xPCI3300 header file 
 *
 *  @par LICENCE
 *  @verbatim
 *  Copyright (C) 2009 - 2014 ADDI-DATA GmbH
 *
 *  ADDI-DATA GmbH
 *  Airpark Business Center
 *  Airport Boulevard B210
 *  77836 Rheinmnster
 *  Germany
 *  Tel: +49(0)7229/1847-0    |  Fax: +49(0)7229/1847-200
 *  http://www.addi-data-com  |  info@addi-data.com
 *  @endverbatim
 **/

#include <windows.h>
#include "define.h"


#ifdef __cplusplus
extern "C"
{
#endif

/** \defgroup PCI3300 xPCI3300 functions
 * @{
 */

/**		\addtogroup General xPCI3300 General functions
 * 		\ingroup PCI3300
 *		@{
 */

/* i_PCI3300_GetNumberOfBoards */
/**
 * Returns the number of xPCI3300 boards present in the system. \n
 * The function fills the value pointed by pb_NbrOfBoards with the number of boards present.
 *
 * @param[out]  pb_NbrOfBoards Value that will be filled with the number of boards
 * @retval  0   No error
 * @retval -1   Error while getting the number of boards
 */
_INT_ i_PCI3300_GetNumberOfBoards(OUT BYTE *pb_NbrOfBoards);


/* i_PCI3300_GetBoardInformation */
/**
 * Gets generic device information.
 *
 * @param[in]   b_BoardIndex Index of the board
 * @param[in]   dw_IdentifierStringSize Size of the identifier parameter buffer
 * @param[out]  pc_Identifier Identifier of the board
 * @param[out]  pdw_UINumber This number is typically a user-perceived slot number, such as a number printed next to the slot on the board, or some other number that makes locating the physical device easier for the user. If the device is on a bus that has no UI number convention, or if the bus driver for the device cannot determine the UI number, this value is 0xFFFFFFFF.
 * @param[out]  pdw_DeviceNumber Device number.
 * @param[out]  pdw_BusNumber Bus number.
 * @param[out]  pdw_BaseAddress0 Base address 0 of the board
 * @param[out]  pdw_BaseAddress1 Base address 1 of the board
 * @param[out]  pdw_BaseAddress2 Base address 2 of the board
 * @param[out]  pdw_BaseAddress3 Base address 3 of the board
 * @param[out]  pb_Interrupt Interrupt line of the board
 * @retval  0   No error
 * @retval -1   Unable to open the board via index
 * @retval -2   Error when calling driver's IOCTL
 */
_INT_ i_PCI3300_GetBoardInformation(IN  BYTE b_BoardIndex,
                                     IN  DWORD dw_IdentifierStringSize,
                                     OUT CHAR  *pc_Identifier,
                                     OUT DWORD *pdw_UINumber,
                                     OUT DWORD *pdw_DeviceNumber,
                                     OUT DWORD *pdw_BusNumber,
                                     OUT DWORD *pdw_BaseAddress0,
                                     OUT DWORD *pdw_BaseAddress1,
                                     OUT DWORD *pdw_BaseAddress2,
                                     OUT DWORD *pdw_BaseAddress3,
                                     OUT BYTE  *pb_Interrupt);

/* i_PCI3300_GetNumberOfModules */
/**
 * Return the number of modules of the board.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[out]  dw_NumberOfModules Board number of modules (2 or 4)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   Failed to read Eeprom
 * @retval -3   Wrong Eeprom functionality
 * @retval -4   Wrong Eeprom number of modules
 * @retval -5   Wrong Eeprom component size
 * @retval -6   Wrong Eeprom number of gain
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_GetNumberOfModules(IN  HANDLE h_DeviceHandle,
                                   OUT DWORD *dw_NumberOfModules);

/* i_PCI3300_OpenBoardViaIndex */
/**
 * Opens the board with the specified index. A handle is returned to the \n
 * user which allows driver functions to be used. \n
 *
 * @param[in]   b_BoardIndex Index of the board to open
 * @param[out]  ph_DeviceHandle Handle of the selected xPCI3300 board
 * @retval  0   No error
 * @retval -1   Unable to get the number of available boards
 * @retval -2   Wrong device index parameter, the parameter is too high
 * @retval -3   Error while getting the device's class
 * @retval -4   Error while getting informations about the device
 * @retval -5   Memory allocation error
 * @retval -6   Get device interface detail error
 * @retval -7   Error while creating communication handle
 * @retval -8   Error in the initialization callback function
 */
_INT_ i_PCI3300_OpenBoardViaIndex(IN BYTE b_BoardIndex, OUT HANDLE *ph_DeviceHandle);


/* i_PCI3300_OpenBoardViaIdentifier */
/**
 * Opens the board with the specified identifier. A handle is returned to the \n
 * user which allows driver functions to be used. \n
 *
 * @param[in]   pc_Identifier Identifier of the board to open
 * @param[out]  ph_DeviceHandle Handle of the xPCI3300 board
 * @retval  0   No Error
 * @retval -1   Unable to get the number of available boards
 * @retval -2   Unable to get the board identifier
 * @retval -3   Unable to open the board
 * @retval -4   Identifier not found
 */
_INT_ i_PCI3300_OpenBoardViaIdentifier(IN CHAR pc_Identifier[], OUT HANDLE *ph_DeviceHandle);


/* i_PCI3300_CloseBoard */
/**
 * Releases the handle to the board. Block the access to the board.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @retval	0   No Error
 * @retval -1   The device handle parameter is wrong
 * @retval -2   Unable to close the device handle
 */
_INT_ i_PCI3300_CloseBoard(HANDLE h_DeviceHandle);


/* i_PCI3300_GetDLLVersion */
/**
 * Gets the version of the DLL.
 *
 * @param[out]  pdw_Version Version. For example: 0x01010910 -> Version 1.01, September 2010
 * @retval  0   No Error
 */
_INT_ i_PCI3300_GetDLLVersion(OUT DWORD *pdw_Version);


/* i_PCI3300_GetDeviceDriverVersion */
/**
 * Gets the version of the device driver.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[out]  pdw_Version Version. For example: 0x01011211 -> Version 1.01, December 2011
 * @retval  0   No Error
 * @retval -1   Error while calling IOCTL function
 */
_INT_ i_PCI3300_GetDeviceDriverVersion(IN HANDLE h_DeviceHandle, OUT DWORD *pdw_Version);

/* i_PCI3300_SetBoardIntRoutine */
/**
 * Registers an interrupt callback function.
 * This callback function will be called each time the board generates an interruption. \n
 * The value of dw_InterruptSource will enable you to determine the source of the interruption:
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   v_FunctionName Address of the callback function
 *		\n	\b h_DeviceHandle: Handle of the board
 *		\n	\b b_FIFOOverflow: If uc_FIFOOverflow != 0, there are too many unprocessed interrupts in the kernel FIFO, and any new information is lost
 *		\n	\b dw_InterruptSource :  Source of the interrupt
 *		\n	\b dw_InterruptMask: Source of the interrupt
 *		\n	\b pdb_Args: Values read during the interrupt.
 *		\n	\b dw_ArgsCount: Number of values read during the interrupt
 *
 * @retval  0   No Error
 * @retval -1   The device handle parameter is wrong
 * @retval -2   An interrupt routine is already installed. Remove it with ResetBoardIntRoutine and retry
 * @retval -3   Error while creating kill event
 * @retval -4   Error while creating interrupt event
 * @retval -5   Error while creating interrupt thread
 * @retval -6   Error while changing the priority class of the thread
 * @retval -7   Error while changing the priority of the thread
 * @retval -8   Error while calling the IOCTL command
 * @retval -9   Error in the interrupt callback function
 */
_INT_ i_PCI3300_SetBoardIntRoutine(IN HANDLE h_DeviceHandle,
                                    IN void (*v_FunctionName)(IN  HANDLE h_DeviceHandle,
                                                              IN  BYTE   b_FifoOverflow,
                                                              IN  DWORD  dw_InterruptSource,
                                                              IN  DWORD  dw_InterruptMask,
                                                              OUT DOUBLE *pdb_Args,
                                                              IN  DWORD  dw_ArgsCount));


/* i_PCI3300_ResetBoardIntRoutine */
/**
 * Deregisters the current interrupt callback function.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @retval  0   No Error
 * @retval -1   The device handle parameter is wrong
 * @retval -2   No interrupt routine installed
 * @retval -3   Error in the reset interrupt callback function
 */
_INT_ i_PCI3300_ResetBoardIntRoutine(IN HANDLE h_DeviceHandle);

/**		@}
 */

/**		\addtogroup Pressure xPCI3300 Pressure functions
 * 		\ingroup PCI3300
 *		@{
 */

/* i_PCI3300_Read1PressureChannel */
/**
 * Returns the analog value (db_ChannelValue) or the pressure value depending of the dw_Dataformat parameter for the selected channel (w_Channel).
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_ModuleIndex Index of the module (0 to number of modules, see i_PCI3300_GetNumberOfModules functions)
 * @param[in]   b_ChannelIndex Channel from the selected module (0..1)
 * @param[in]   b_Gain Channel gain (0..7)
 *                    \li 0: Gain 1
 *                    \li 1: Gain 2
 *                    \li 2: Gain 4
 *                    \li 3: Gain 8
 *                    \li 4: Gain 16
 *                    \li 5: Gain 32
 *                    \li 6: Gain 64
 *                    \li 7: Gain 128
 * @param[in]   dw_DataFormat Format of the db_ChannelValue
 *                    \li 0: db_ChannelValue in V (db_SensorOffset and db_SensorSensibility are not used)
 *                    \li 1: db_ChannelValue in Unit (depends on db_SensorSensibility unit)
 * @param[in]   db_SensorOffset Sensor offset voltage (mV). Refer to the sensor documentation
 * @param[in]   db_SensorSensibility Sensor sensibility (mV/V/unit). Refer to the sensor documentation (unit depends on the sensor)
 * @param[in]   dw_ConvertingTime Converting time (20..255)
 * @param[in]   b_Option Not used, must be 0
 * @param[out]  db_ChannelValue Value of the channel (See dw_DataFormat for the unit)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_ModuleIndex parameter is wrong
 * @retval -3   The b_ChannelMask parameter is wrong
 * @retval -4   The b_Gain parameter is wrong
 * @retval -5   The dw_ConvertingTime parameter is wrong
 * @retval -6   The b_Interrupt parameter is wrong
 * @retval -7   No module information, please call i_PCI3300_GetNumberOfModules function.
 * @retval -8   An acquisition is already started
 * @retval -9   Timeout occurs when writing parameter
 * @retval -10  Timeout occurs when getting channel value
 * @retval -11  dw_DataFormat parameter is wrong must be 0 or 1 
 * @retval -12  Failed to get board information
 * @retval -13  Failed to convert analog value to pressure (db_SensorSensibility must not be 0)
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Read1PressureChannel(IN  HANDLE h_DeviceHandle,
                                    IN  BYTE b_ModuleIndex,
                                    IN  BYTE b_ChannelIndex,
                                    IN  BYTE b_Gain,
									IN  DWORD dw_DataFormat,
                                    IN  DOUBLE db_SensorOffset,
                                    IN  DOUBLE db_SensorSensibility,
                                    IN  DWORD dw_ConvertingTime,
                                    IN  BYTE b_Option,
                                    OUT DOUBLE *db_ChannelValue);

/* i_PCI3300_InitAndStartScanModuleAcquisition */
/**
 * Initializes the selected analog-input module for a scan acquisition and start it.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_ModuleIndex Index of the module (0 to number of modules, see i_PCI3300_GetNumberOfModules functions)
 * @param[in]   dw_ConvertingTime Converting time in Hz
 *                              \li 20: 20 Hz
 *                              \li 40: 40 Hz
 *                              \li 80: 80 Hz
 *                              \li 160: 160 Hz
 * @param[in]   b_DelayMode Delay mode selection
 *                              \li 0: No delay used
 *                              \li 1: Delay used (0..1)
 * @param[in]   w_DelayTime Delay between two acquisitions (the acquisition time is included). (0..1023) \n
 *                          Must be greater than 5 * period (At 20 Hz Delay must be greater than 5 * 1/20 = 0.25s)
 * @param[in]   w_DelayUnit Delay unit
 *                               \li 0: ms
 *                               \li 1: s
 * @param[in]   b_AcquisitionMode Specify if you want to use continuous acquisition or not.
 *                               \li 0: Single scan
 *                               \li 1: Continuous scan
 * @param[in]   b_ExtTriggerMode Trigger configuration (the trigger starts all scan) 
 * 								 \li 0: Trigger disabled
 * 								 \li 1: Trigger on rising edge
 * 								 \li 2: Trigger on falling edge
 * 								 \li 3: Trigger on both edges
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_ModuleIndex parameter is wrong
 * @retval -3   The b_DelayMode parameter is wrong
 * @retval -4   The w_DelayTime parameter is wrong
 * @retval -5   The w_DelayUnit parameter is wrong
 * @retval -6   The b_AcquisitionMode parameter is wrong
 * @retval -7   The b_ExtTriggerMode parameter is wrong
 * @retval -8   No module information, please call i_PCI3300_GetNumberOfModules function.
 * @retval -9   An acquisition is already started
 * @retval -10  The delay time is too small, must be greater than one acquistion (5 * period)
 * @retval -11  Failed to init channel
 * @retval -12  Timeout occurs when writing parameter
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_InitAndStartScanModuleAcquisition(IN  HANDLE h_DeviceHandle,
                                                  IN  BYTE b_ModuleIndex,
                                                  IN  DWORD dw_ConvertingTime,
                                                  IN  BYTE b_DelayMode,
                                                  IN  WORD w_DelayTime,
                                                  IN  WORD w_DelayUnit,
                                                  IN  BYTE b_AcquisitionMode,
                                                  IN  BYTE b_ExtTriggerMode);


/* i_PCI3300_StopScanModuleAcquisition */
/**
 * Stop a scan acquisition on the selected analog input module.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_ModuleIndex Index of the module Index of the module (0 to maximum module number (see i_PCI3300_GetNumberOfModules functions))
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_ModuleIndex parameter is wrong
 * @retval -3   No acquisition started on this module
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_StopScanModuleAcquisition(IN  HANDLE h_DeviceHandle,
                                          IN  BYTE b_ModuleIndex);


/* i_PCI3300_GetModuleAcquisitionStatus */
/**
 * Get the status of the module to see if an acquisition is started.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_ModuleIndex Index of the module Index of the module (0 to maximum module number (see i_PCI3300_GetNumberOfModules functions))
 * @param[out]  dw_Status Status of the acquisition
 *                        \li 0: No acquisition running
 *                        \li 1: Read acquisition started
 *                        \li 2: Scan acquisition status
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_ModuleIndex parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_GetModuleAcquisitionStatus(IN  HANDLE h_DeviceHandle,
                                           IN  BYTE b_ModuleIndex,
                                           OUT DWORD *dw_Status);


/* i_PCI3300_ConvertAnalogToPressure */
/**
 * Bridge tension is 5 V\n
 * Compute the pressure value with the following formula (Unit depends on sensor):\n
 * db_PressureValue<Unit> = (db_AnalogValue<mV> + db_SensorOffset<mV>) / (db_SensorSensibility<mV/V/Unit> * 5<V>)
 *
 * @param[in] db_AnalogValue "Analog value to convert in mV"
 * @param[in] db_SensorOffset "Sensor offset in mV"
 * @param[in] db_SensorSensibility "Sensor sensibility in mV/V/Unit"
 * @param[out] db_PressureValue "Computed pressure value (unit depends on sensor)"
 *
 * @retval  0   No error
 * @retval -1   db_SensorSensibility cannot be 0.0
 */
_INT_ i_PCI3300_ConvertAnalogToPressure(DOUBLE db_AnalogValue, DOUBLE db_SensorOffset, DOUBLE db_SensorSensibility, DOUBLE *db_PressureValue);
		
/**		@}
 */

/**		\addtogroup DInputs xPCI3300 Digital inputs functionality
 * 		\ingroup PCI3300
 *		@{
 */

/* i_PCI3300_Read1DigitalInput */
/**
 * Read 1 digital input of the board
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Channel Channel to read (0..3)
 * @param[out]  b_Value Status of the selected digital input
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_Channel parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Read1DigitalInput(IN  HANDLE h_DeviceHandle,
                                  IN  BYTE b_Channel,
                                  OUT BYTE *b_Value);


/* i_PCI3300_Read4DigitalInputs */
/**
 * Read 4 digital inputs of the board
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[out]  b_Value Status of the digital inputs
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Read4DigitalInputs(IN  HANDLE h_DeviceHandle,
                                   OUT BYTE *b_Value);

/**		@}
 */

/**		\addtogroup DOutputs xPCI3300 Digital outputs functionality
 * 		\ingroup PCI3300
 *		@{
 */

/* i_PCI3300_Set1DigitalOutputOn */
/**
 * Set 1 digital output of the board to high level
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Channel Channel to set (0..2)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_Channel parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Set1DigitalOutputOn(IN  HANDLE h_DeviceHandle,
                                    IN  BYTE b_Channel);


/* i_PCI3300_Set1DigitalOutputOff */
/**
 * Set 1 digital output of the board to low level
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Channel Channel to set (0..2)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_Channel parameter is wrong
 * @retval -3   The output memory is not set can't reset output (see function i_PCI3300_SetDigitalOutputMemoryOn)
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Set1DigitalOutputOff(IN  HANDLE h_DeviceHandle,
                                     IN  BYTE b_Channel);


/* i_PCI3300_Set3DigitalOutputsOn */
/**
 * Set 3 digital outputs of the board to high level
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Mask Channels to set (0x0 to 0x7)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Set3DigitalOutputsOn(IN  HANDLE h_DeviceHandle,
                                     IN  BYTE b_Mask);


/* i_PCI3300_Set3DigitalOutputsOff */
/**
 * Set 3 digital outputs of the board to low level
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Mask Channels to reset (0x0 to 0x7)
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The output memory is not set can't reset output (see function i_PCI3300_SetDigitalOutputMemoryOn)
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Set3DigitalOutputsOff(IN  HANDLE h_DeviceHandle,
                                      IN  BYTE b_Mask);


/* i_PCI3300_SetDigitalOutputMemoryOn */
/**
 * Enable the digital output memory
 * When digital output memory is enabled, the active channels stay active until calling i_PCI3300_Set1DigitalOutputOff or i_PCI3300_Set4DigitalOutputOff.
 * When digital output memory is disabled, the active channels are reset at each call of i_PCI3300_Set1DigitalOutputOn or i_PCI3300_Set4DigitalOutputOn.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_SetDigitalOutputMemoryOn(IN  HANDLE h_DeviceHandle);


/* i_PCI3300_SetDigitalOutputMemoryOff */
/**
 * Disable the digital output memory
 * When digital output memory is enabled, the active channels stay active until calling i_PCI3300_Set1DigitalOutputOff or i_PCI3300_Set3DigitalOutputOff.
 * When digital output memory is disabled, the active channels are reset at each call of i_PCI3300_Set1DigitalOutputOn or i_PCI3300_Set3DigitalOutputOn.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_SetDigitalOutputMemoryOff(IN  HANDLE h_DeviceHandle);


/* i_PCI3300_Get1DigitalOutputStatus */
/**
 * Get the status of 1 digital output of the board.
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[in]   b_Channel Channel to get (0..2)
 * @param[out]  b_OutputsStatus Status of the digital output
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -2   The b_Channel parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Get1DigitalOutputStatus(IN  HANDLE h_DeviceHandle,
                                        IN  BYTE b_Channel,
                                        OUT BYTE *b_OutputsStatus);


/* i_PCI3300_Get3DigitalOutputsStatus */
/**
 * Get the status of all the digital outputs of the board
 *
 * @param[in]   h_DeviceHandle Handle of the board
 * @param[out]  b_OutputsStatus Status of the digital outputs
 * @retval  0   No error
 * @retval -1   The h_DeviceHandle parameter is wrong
 * @retval -100 An error occurred while calling the IOCTL function
 */
_INT_ i_PCI3300_Get3DigitalOutputsStatus(IN  HANDLE h_DeviceHandle,
                                         OUT BYTE *b_OutputsStatus);

/**		@}
 */
#ifdef __cplusplus
}
#endif
