/** @file ReadPressureChannel.c
 *
 *  @authors S. Beuque
 *  @date    Creation: 06.10.2014
 *
 *  @par DESCRIPTION
 *  This sample shows how to init and read pressure channel. 
 *
 *  @par LICENCE
 *  @verbatim
 *  Copyright (C) 2009 - 2014 ADDI-DATA GmbH
 *
 *  ADDI-DATA GmbH
 *  Airpark Business Center
 *  Airport Boulevard B210
 *  77836 Rheinmnster
 *  Germany
 *  Tel: +49(0)7229/1847-0    |  Fax: +49(0)7229/1847-200
 *  http://www.addi-data-com  |  info@addi-data.com
 *  @endverbatim
 **/

#include "StdAfx.h"

int _tmain(int argc, _TCHAR *argv[])
{
	INT    i_ReturnValue        = 0;
	HANDLE h_BoardHandle        = NULL;
	BYTE   b_ModuleIndex        = 0;
	BYTE   b_ChannelIndex       = 0;
	BYTE   b_Gain               = 0;
	DWORD  dw_ConvertingTime    = 0;
	DOUBLE db_ChannelValue      = 0;
	DWORD  dw_NumberOfModules   = 0;
	DOUBLE db_SensorSensibility = 0;
	DOUBLE db_OffsetVoltage     = 0;
	DWORD  dw_DataFormat        = 0;
	DWORD  dw_Status            = 0;

	UNREFERENCED_PARAMETER(argc);
	UNREFERENCED_PARAMETER(argv);

	printf("/****************************************************************************\\\n"
	       "|*                        Read1PressureChannel sample                       *|\n"
	       "|****************************************************************************|\n"
	       "|* This sample shows how to init and read 1 pressure channel.               *|\n"
	       "\\****************************************************************************/\n\n");

	i_ReturnValue = i_Utils_OpenBoard(&h_BoardHandle);
	if (i_ReturnValue != 0)
	{
		fprintf(stderr, "i_Utils_OpenBoard failed: %d\n", i_ReturnValue);
		_getch();
		return i_ReturnValue;
	}

	i_ReturnValue = i_PCI3300_GetNumberOfModules(h_BoardHandle, &dw_NumberOfModules);
	if (i_ReturnValue != 0)
	{
		fprintf(stderr, "i_PCI3300_GetBoardEepromInfo failed: %d\n", i_ReturnValue);
		goto sample_end;
	}

	b_ModuleIndex  = (BYTE)ll_Utils_ReadInteger(0, dw_NumberOfModules - 1, "Which module do you want to read (0 to %d)", dw_NumberOfModules - 1);

	b_ChannelIndex = (BYTE)ll_Utils_ReadInteger(0, 1, "Enter the channel to be read (0 to 1)\n");

	b_Gain = (BYTE)ll_Utils_ReadInteger(0, 7, "Choose the gain :\n\t0: Gain 1"
															   "\n\t1: Gain 2"
															   "\n\t2: Gain 4"
															   "\n\t3: Gain 8"
															   "\n\t4: Gain 16"
															   "\n\t5: Gain 32"
															   "\n\t6: Gain 64"
															   "\n\t7: Gain 128");

	do
	{
		dw_ConvertingTime = (DWORD)ll_Utils_ReadInteger(20, 160, "Enter the converting time in Hz (20 Hz, 40 Hz, 80 Hz or 160 Hz)");
	}while((dw_ConvertingTime != 20) && (dw_ConvertingTime != 40) && (dw_ConvertingTime != 80) && (dw_ConvertingTime != 160));

	dw_DataFormat = (BYTE)ll_Utils_ReadInteger(0, 1, "Do you want to get:\n\t0: The analog value in mV\n\t1: The pressure value");

	if (dw_DataFormat != 0)
	{
		printf("Enter the sensor offset voltage in mV: \n");
		scanf_s("%lf", &db_OffsetVoltage);

		do
		{
			printf("Enter the sensor sensibility in mV/V/unit (Unit depends on sensor): \n");
			scanf_s("%lf", &db_SensorSensibility);
		}while(db_SensorSensibility == 0.0);
	}

	do
	{
		if (dw_Status == 0)
		{
			i_ReturnValue = i_PCI3300_Read1PressureChannel(h_BoardHandle,
				b_ModuleIndex,
				b_ChannelIndex,
				b_Gain,
				dw_DataFormat,
				db_OffsetVoltage,
				db_SensorSensibility,
				dw_ConvertingTime,
				0,
				&db_ChannelValue);
			if (i_ReturnValue != 0)
				{ printf("i_PCI3300_ReadPressureChannel error %d\n", i_ReturnValue); goto sample_end; }
		}

		printf("\rdw_ChannelValue = %lf %s", db_ChannelValue, ((dw_DataFormat == 0) ? "V" : ""));

		Sleep(100);
	}while(!_kbhit());

sample_end:
	i_ReturnValue = i_PCI3300_CloseBoard(h_BoardHandle);
	if (i_ReturnValue != 0)
	{
		fprintf(stderr, "i_PCI3300_CloseBoard failed: %d\n", i_ReturnValue);
		_getch();
		return i_ReturnValue;
	}
	printf("i_PCI3300_CloseBoard OK\n");

	printf("\nPress any key to exit the sample...\n");
	_getch();
	return i_ReturnValue;
}
