/** @file Utils.c
 *
 *  @authors S. Beuque
 *  @date    Creation: 06.10.2014
 *
 *  @par DESCRIPTION
 *  Miscellaneous sample utilities 
 *
 *  @par LICENCE
 *  @verbatim
 *  Copyright (C) 2009 - 2014 ADDI-DATA GmbH
 *
 *  ADDI-DATA GmbH
 *  Airpark Business Center
 *  Airport Boulevard B210
 *  77836 Rheinmnster
 *  Germany
 *  Tel: +49(0)7229/1847-0    |  Fax: +49(0)7229/1847-200
 *  http://www.addi-data-com  |  info@addi-data.com
 *  @endverbatim
 **/

#include "StdAfx.h"

DWORD _convertToPressure = 0;
DOUBLE _sensorOffset[2];
DOUBLE _sensorSensibility[2];

/**
*  Sample interrupt routine to be registered with the i_PCI3300_SetBoardIntRoutine function
**/
void v_Utils_InterruptRoutine(HANDLE  h_DeviceHandle,
                              BYTE    b_FifoOverflow,
                              DWORD   dw_InterruptSource,
                              DWORD   dw_InterruptMask,
                              DOUBLE* pdb_Args,
                              DWORD   dw_ArgsCount)
{
	UNREFERENCED_PARAMETER(h_DeviceHandle);
	UNREFERENCED_PARAMETER(dw_InterruptMask);
	UNREFERENCED_PARAMETER(dw_ArgsCount);

	/* Check the if an interrupt fifo overflow has occurred */
	if (b_FifoOverflow != 0)
	{
		printf("Error: The interrupt FIFO buffer has overflowed!"
		       "This means that your software does not handle the interrupts fast enough.\n");
	}

	/* Switch through the possible interrupt sources */

	/* Read interruption? */
	if ((dw_InterruptSource & 0x1) != 0)
	{
		printf("\nRead interrupt!\n");
	}

	/* Scan interruption? */
	if ((dw_InterruptSource & 0x2) != 0)
	{
		DWORD i;
		printf("\nScan interrupt!\n");
		for (i = 0; i < 2; i++)
		{
			if (_convertToPressure != 0)
			{
				if (i_PCI3300_ConvertAnalogToPressure(pdb_Args[i], _sensorOffset[i], _sensorSensibility[i], &pdb_Args[i]) != 0)
				{
					printf("i_PCI3300_ConvertAnalogToPressure db_SensorSensibility cannot be 0.0\n");
					continue;
				}
			}

			printf("Channel %i = %lf %s\n", i, pdb_Args[i], (_convertToPressure == 0) ? "V" : "" );
		}
		printf("\n");
	}

	printf("\n");
}

/**
*  Helper function, used to open a specific board
**/
_INT_ i_Utils_OpenBoard(HANDLE *ph_BoardHandle)
{
	INT   ret;
	DWORD version;
	BYTE  boardNbr;
	BYTE  i;

	if (i_PCI3300_GetDLLVersion(&version) == 0)
		printf("DLL version: %04x-%04x\n", (version >> 16) & 0xFFFF, version & 0xFFFF);

	ret = i_PCI3300_GetNumberOfBoards(&boardNbr);
	if (ret != 0)
	{
		fprintf(stderr, "i_PCI3300_GetNumberOfBoards failed: %d\n", ret);
		return ret;
	}

	printf("%d board%s found.\n\n", boardNbr, boardNbr > 1 ? "s were" : " was");
	if (boardNbr < 1)
		return -1;

	for (i = 0; i < boardNbr; i++)
	{
		CHAR  identifier[50];
		DWORD uiNumber;
		DWORD deviceNumber;
		DWORD busNumber;
		DWORD baseAddress[4];
		BYTE  interrupt;

		ret = i_PCI3300_GetBoardInformation(i, sizeof(identifier), identifier, &uiNumber, &deviceNumber, &busNumber, &baseAddress[0], &baseAddress[1], &baseAddress[2], &baseAddress[3], &interrupt);
		if (ret != 0)
		{
			fprintf(stderr, "i_PCI3300_GetBoardInformation failed for board %d: %d\n", i, ret);
			continue;
		}

		printf("Board %d information:\n", i);
		printf("  Identifier: %s\n", identifier);
		printf("  UI number: 0x%x; Device number: 0x%x; Bus number: 0x%x\n", uiNumber, deviceNumber, busNumber);
		printf("  Base addresses: 0x%x / 0x%x / 0x%x / 0x%x\n", baseAddress[0], baseAddress[1], baseAddress[2], baseAddress[3]);
		printf("  Interrupt: %d\n", interrupt);
		printf("\n");
	}

	if (ll_Utils_ReadInteger(0, 1, "Should the board be opened via index (0) or via identifier (1)?") == 0)
	{
		ret = i_PCI3300_OpenBoardViaIndex((BYTE)ll_Utils_ReadInteger(0, boardNbr - 1, "Enter the index of the board (0 to %d):", boardNbr - 1), ph_BoardHandle);
		if (ret != 0)
		{
			fprintf(stderr, "i_PCI3300_OpenBoardViaIndex failed: %d\n", ret);
			return ret;
		}
	}
	else
	{
		CHAR identifier[50];
		printf("Enter the board identifier: ");
		scanf_s("%s", identifier);
		ret = i_PCI3300_OpenBoardViaIdentifier(identifier, ph_BoardHandle);
		if (ret != 0)
		{
			fprintf(stderr, "i_PCI3300_OpenBoardViaIdentifier failed: %d\n", ret);
			return ret;
		}
	}

	if (i_PCI3300_GetDeviceDriverVersion(*ph_BoardHandle, &version) == 0)
		printf("Device driver version: %04x-%04x\n", (version >> 16) & 0xFFFF, version & 0xFFFF);
	printf("\n");

	return ret;
}

/**
*  Prompts the user to enter a value within specified bounds
**/
_LONGLONG_ ll_Utils_ReadInteger(LONGLONG l_MinValue, LONGLONG l_MaxValue, PCHAR pc_Description, ...)
{
	va_list args;
	LONGLONG readValue;

	va_start(args, pc_Description);
	do
	{
		vfprintf(stdout, pc_Description, args);
		fprintf(stdout, " ");
		fflush(stdout);
		scanf_s("%llu", &readValue);
	}
	while (readValue < l_MinValue || readValue > l_MaxValue);
	va_end(args);
	return readValue;
}

/**
*  Display a sort of progress bar
**/
_VOID_ v_Utils_DisplayRunningStatus()
{
	static int charNbr = 0;
	int i = 0;

	for (i = 0; i < 5; i++)
		printf(i < charNbr ? "." : " ");
	printf("\r");

	charNbr = (charNbr + 1) % 5;
}


