/** @file PCI3300.cs
 *
 *  @authors S. Beuque
 *  @date    Creation: 06.10.2014
 *
 *  @par DESCRIPTION
 *  PCI3300 .NET interface functions 
 *
 *  @par LICENCE
 *  @verbatim
 *  Copyright (C) 2009 - 2014 ADDI-DATA GmbH
 *
 *  ADDI-DATA GmbH
 *  Airpark Business Center
 *  Airport Boulevard B210
 *  77836 Rheinmnster
 *  Germany
 *  Tel: +49(0)7229/1847-0    |  Fax: +49(0)7229/1847-200
 *  http://www.addi-data-com  |  info@addi-data.com
 *  @endverbatim
 **/

using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;

namespace PCI3300_NET
{
#region "User declarations"
#endregion

    public class PCI3300
    {
        private int index;
        private IntPtr handle;

		private DllImport.InterruptFunctionEventHandler interruptFunctionHandler;

#region "Event Args and Handlers"
        public delegate void ReadEventHandler(object source);
        public event ReadEventHandler ReadEventReceived;

        public delegate void ScanEventHandler(IntPtr handle, Byte overflow, UInt32 mask, double[] args, UInt32 argsSize);
        public event ScanEventHandler ScanEventReceived;

#endregion
#region "Constructor/Destructor"
        /// <summary>
        /// Class constructor
        /// </summary>
        private PCI3300()
        {
            this.index = -1;
            this.handle = IntPtr.Zero;

            // Create the event args (once by board)

            // Create the interrupt handler (once by board)
            this.interruptFunctionHandler = new DllImport.InterruptFunctionEventHandler(InterruptFunction);

            // Tell the garbage collector to keep the pointer
            GC.KeepAlive(this.interruptFunctionHandler);

        }

        /// <summary>
        /// Class destructor
        /// </summary>
        ~PCI3300()
        {
            DllImport.i_PCI3300_CloseBoard(this.handle);
            this.index = -1;
            this.handle = IntPtr.Zero;

        }
#endregion
#region "Utils"
        /// <summary>
        /// Creates an instance of a PCI3300 board
        /// </summary>
        /// <param name="index">The index of the board you want to open</param>
        /// <returns>null on error, instance of PCI3300 on success</returns>
        public static PCI3300 OpenBoard(int index)
        {
            PCI3300 boardInstance;
            int nbBoard;
            IntPtr handle = IntPtr.Zero;

            // Get the number of boards
			nbBoard = GetBoardCount();
            if (nbBoard == -1)
            {
                Console.WriteLine("GetBoardCount failed!");
                return null;
            }

            // Check that the index is in the range
            if (index < 0 || index > nbBoard)
            {
                Console.WriteLine("Invalid index!");
                return null;
            }

            // Try to open the board
            if (DllImport.i_PCI3300_OpenBoardViaIndex((Byte)index, out handle) != 0)
            {
                Console.WriteLine("i_PCI3300_OpenBoardViaIndex failed!");
                return null;
            }

            // Create a new instance of the board
            boardInstance = new PCI3300();

            // Set its parameters
            boardInstance.index = index;
            boardInstance.handle = handle;

            // Return the instance
            return boardInstance;
        }

        /// <summary>
        /// Returns the number of PCI3300 boards plugged in the computer
        /// </summary>
        /// <returns>The number of boards</returns>
        public static int GetBoardCount()
        {
            int ret;
            byte nbBoard = 0;

            // Call the DLL function
            ret = DllImport.i_PCI3300_GetNumberOfBoards(out nbBoard);
            if (ret != 0)
				return -1;

             // Success
            return (int)nbBoard;
        }

        /// <summary>
        /// Returns the description of the last error that occurred
        /// </summary>
        /// <returns>A string describing the last error that occurred</returns>
        public string GetLastError()
        {
            // Call the DLL function
            return Marshal.PtrToStringAnsi(DllImport.pc_PCI3300_GetLastError(this.handle));
        }
#endregion
#region "Interrupt handling"
        /// <summary>
        /// Signals to all the listeners that an Read event occurred.
        /// </summary>
        protected virtual void OnReadEventReceived()
		{
            if (this.ReadEventReceived != null)
                this.ReadEventReceived(this);
        }

        /// <summary>
        /// Signals to all the listeners that an Scan event occurred.
        /// </summary>
        protected virtual void OnScanEventReceived(IntPtr handle, Byte overflow, UInt32 mask, double[] args, UInt32 argsSize)
		{
            if (this.ScanEventReceived != null)
                this.ScanEventReceived(handle, overflow, mask, args, argsSize);
        }

        /// <summary>
        /// Interrupt function - Called each time the board has generated an interrupt
        /// </summary>
        private void InterruptFunction(IntPtr handle, Byte overflow, UInt32 source, UInt32 mask, IntPtr args, UInt32 argsSize)
        {
            byte[] data = new byte[argsSize * sizeof(double)];
            double[] interruptArgs = new double[argsSize];

            Marshal.Copy(args, data, 0, (int)argsSize * sizeof(double));
            for (int i = 0; i < argsSize; i++)
                interruptArgs[i] = BitConverter.ToDouble(data, i * sizeof(double));

            // Read
            if ((source & 0x1) != 0)
            {
                // Signal the event
                this.OnReadEventReceived();
            }

            // Scan
            if ((source & 0x2) != 0)
            {
                // Signal the event
                this.OnScanEventReceived(handle, overflow, mask, interruptArgs, argsSize);
            }

        }

        /// <summary>
        /// Set the board interrupt routine
        /// </summary>
        public int SetBoardIntRoutine()
        {
            return (int)DllImport.i_PCI3300_SetBoardIntRoutine(this.handle, interruptFunctionHandler);
        }

        /// <summary>
        /// Reset the board interrupt routine
        /// </summary>
        public int ResetBoardIntRoutine()
        {
            return (int)DllImport.i_PCI3300_ResetBoardIntRoutine(this.handle);
        }

#endregion

#region "PCI3300"

#endregion
#region "Pressure"

		/// <summary>
		/// Initializes the selected pressure channel.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="channel">Channel index to be initialized</param>
		public int InitPressureChannel(Byte channel)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_InitPressureChannel(this.handle, channel);
			return returnValue;
		}

		/// <summary>
		/// Returns the digital value (pdw_ChannelValue) of the selected channel (w_Channel). To obtain the real pressure value, you must call the i_PCI3300_ConvertDigitalToRealPressureValue(...) function.
		/// dw_ConvertingTime and b_ConvertingTimeUnit determine the converting time.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="moduleIndex">Module index of the board</param>
        /// <param name="channelMask">Channel index for the selected module (0: channel 0, 1: channel 1)</param>
        /// <param name="dataFormat">To display pressure value(0) or analog value (0)</param>
		/// <param name="gain">Channel gain</param>
        /// <param name="sensorOffset">Sensor offset voltage (mV). Refer to the sensor documentation</param>
		/// <param name="sensorSensibility">Sensor sensibility (mV/V/bar). Refer to the sensor documentation</param>
		/// <param name="convertingTime">Converting time</param>
		/// <param name="option">Not used, must be 0</param>
		/// <param name="channelValue">Value of the channel only if interrupt is disabled, else channel value return with interrupt</param>
        public int Read1PressureChannel(Byte moduleIndex, Byte channelIndex, Byte gain, UInt32 dataFormat, double sensorOffset, double sensorSensibility, UInt32 convertingTime, Byte interrupt, out double channelValue)
        {
            int returnValue;
            returnValue = DllImport.i_PCI3300_Read1PressureChannel(this.handle, moduleIndex, channelIndex, gain, dataFormat, sensorOffset, sensorSensibility, convertingTime, interrupt, out channelValue);
            return returnValue;
        }

		/// <summary>

		/// <summary>
		/// Initialize the selected analog-input module for a scan acquisition.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="moduleIndex">Index of the module (0 to 3)</param>
		/// <param name="convertingTime">Converting time in Hz (20, 40, 80, 160)</param>
		/// <param name="delayMode">0: no delay used, 1: delay used</param>
		/// <param name="delayTime">Delay between two acquisition (the acquisition time is included). (0..1023)</param>
		/// <param name="delayUnit">Delay unit 0: ms, 1: s</param>
		/// <param name="acquisitionMode">Specify if you want to use continuous acquisition or not 0: single scan, 1: continuous scan</param>
		/// <param name="extTriggerMode">Trigger configuration</param>
		public int InitAndStartScanModuleAcquisition(Byte moduleIndex, UInt32 convertingTime, Byte delayMode, UInt16 delayTime, UInt16 delayUnit, Byte acquisitionMode, Byte extTriggerMode)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_InitAndStartScanModuleAcquisition(this.handle, moduleIndex, convertingTime, delayMode, delayTime, delayUnit, acquisitionMode, extTriggerMode);
			return returnValue;
		}

		/// <summary>
		/// Stop a scan acquisition on the selected analog input module.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="moduleIndex">Index of the module (0 to 3)</param>
		public int StopScanModuleAcquisition(Byte moduleIndex)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_StopScanModuleAcquisition(this.handle, moduleIndex);
			return returnValue;
		}

		/// <summary>
		/// Get the status of the module to see if an acquisition is started.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="moduleIndex">Index of the module (0 to 3)</param>
		/// <param name="status">Status of the acquisition (0: No acquisition running, 1: Read acquisition started, 2: Scan acquisition status</param>
		public int GetModuleAcquisitionStatus(Byte moduleIndex, out UInt32 status)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_GetModuleAcquisitionStatus(this.handle, moduleIndex, out status);
			return returnValue;
		}

        /// <summary>
        /// 
        /// </summary>
        /// <param name="analogValue">Analog value to convert in mV</param>
        /// <param name="sensorOffset">Sensor offset in mV</param>
        /// <param name="sensorSensibility">Sensor sensibility in mV</param>
        /// <param name="pressureValue">Pressure value in mV</param>
        public int ConvertAnalogToPressure(double analogValue, double sensorOffset, double sensorSensibility, out double pressureValue)
        {
			int returnValue;
            returnValue = DllImport.i_PCI3300_ConvertAnalogToPressure(analogValue, sensorOffset, sensorSensibility, out pressureValue);
			return returnValue;
		}

#endregion
#region "DInputs"

		/// <summary>
		/// Read 1 digital input of the board
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="channel">Channel to read</param>
		/// <param name="value">Status of the selected digital input</param>
		public int Read1DigitalInput(Byte channel, out Byte value)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Read1DigitalInput(this.handle, channel, out value);
			return returnValue;
		}

		/// <summary>
		/// Read 4 digital input of the board
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="value">Status of the digital inputs</param>
		public int Read4DigitalInputs(out Byte value)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Read4DigitalInputs(this.handle, out value);
			return returnValue;
		}

#endregion
#region "DOutputs"

		/// <summary>
		/// Set 1 digital output of the board to high level
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="channel">Channel to set</param>
		public int Set1DigitalOutputOn(Byte channel)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Set1DigitalOutputOn(this.handle, channel);
			return returnValue;
		}

		/// <summary>
		/// Set 1 digital output of the board to low level
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="channel">Channel to set</param>
		public int Set1DigitalOutputOff(Byte channel)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Set1DigitalOutputOff(this.handle, channel);
			return returnValue;
		}

		/// <summary>
		/// Set 3 digital outputs of the board to high level
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="mask">Channels to set (0x0 to 0x7)</param>
		public int Set3DigitalOutputsOn(Byte mask)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Set3DigitalOutputsOn(this.handle, mask);
			return returnValue;
		}

		/// <summary>
		/// Set 3 digital outputs of the board to low level
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="mask">Channels to reset (0x0 to 0x7)</param>
		public int Set3DigitalOutputsOff(Byte mask)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Set3DigitalOutputsOff(this.handle, mask);
			return returnValue;
		}

		/// <summary>
		/// Enable the digital output memory
		/// When digital output memory is enabled, the active channels stay active until calling i_PCI3300_Set1DigitalOutputOff or i_PCI3300_Set4DigitalOutputOff.
		/// When digital output memory is disabled, the active channels are reset at each call of i_PCI3300_Set1DigitalOutputOn or i_PCI3300_Set4DigitalOutputOn.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		public int SetDigitalOutputMemoryOn()
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_SetDigitalOutputMemoryOn(this.handle);
			return returnValue;
		}

		/// <summary>
		/// Disable the digital output memory
		/// When digital output memory is enabled, the active channels stay active until calling i_PCI3300_Set1DigitalOutputOff or i_PCI3300_Set4DigitalOutputOff.
		/// When digital output memory is disabled, the active channels are reset at each call of i_PCI3300_Set1DigitalOutputOn or i_PCI3300_Set4DigitalOutputOn.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		public int SetDigitalOutputMemoryOff()
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_SetDigitalOutputMemoryOff(this.handle);
			return returnValue;
		}

		/// <summary>
		/// Get the status of 1 digital output of the board.
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="channel">Channel to get (0 to 2)</param>
		/// <param name="outputsStatus">Status of the digital output</param>
		public int Get1DigitalOutputStatus(Byte channel, out Byte outputsStatus)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Get1DigitalOutputStatus(this.handle, channel, out outputsStatus);
			return returnValue;
		}

		/// <summary>
		/// Get the status of all the digital outputs of the board
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="outputsStatus">Status of the digital output</param>
		public int Get3DigitalOutputsStatus(out Byte outputsStatus)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_Get3DigitalOutputsStatus(this.handle, out outputsStatus);
			return returnValue;
		}

#endregion
#region "board"

		/// <summary>
		/// Get the board number of modules
		/// </summary>
		/// <param name="deviceHandle">Handle of the board</param>
		/// <param name="numberOfModules">Board number of modules</param>
        public int GetNumberOfModules(out UInt32 numberOfModules)
		{
			int returnValue;
			returnValue = DllImport.i_PCI3300_GetNumberOfModules(this.handle, out numberOfModules);
			return returnValue;
		}

#endregion
#region "User"
#endregion
    }
}
