using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Globalization;

namespace Read1Pressure
{
    public partial class Read1Pressure : Form
    {
        /// <summary>
        /// Handle to the first board
        /// </summary>
        private PCI3300_NET.PCI3300 board;

        /// <summary>
        /// Number of modules of the board
        /// </summary>
        private uint _numberOfModules;

        /// <summary>
        /// Constructor of the form
        /// </summary>
        public Read1Pressure()
        {
            InitializeComponent();
        }

        /// <summary>
        /// Called at the first load
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Read1Pressure_Load(object sender, EventArgs e)
        {
            try
            {
                int returnValue = 0;

                // Open the first board of the system
                this.board = PCI3300_NET.PCI3300.OpenBoard(0);
                // Error handling
                if (this.board == null)
                {
                    MessageBox.Show("Error OpenBoard returned null");
                    this.Close();
                    return;
                }

                // Get the board number of modules
                returnValue = this.board.GetNumberOfModules(out _numberOfModules);
                if (returnValue != 0)
                {
                    MessageBox.Show("GetNumberOfModules error : " + returnValue.ToString());
                    this.Close();
                    return;
                }

                for (int i = 0; i < _numberOfModules; i++)
                    moduleIndexComboBox.Items.Add(i);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString());
                return;
            }

            Thread.CurrentThread.CurrentCulture = new CultureInfo("en-US");

            // Initialize the combo boxes
            this.moduleIndexComboBox.SelectedIndex = 0;
            this.channelIndexComboBox.SelectedIndex = 0;
            this.gainComboBox.SelectedIndex = 0;
            this.convertingTimeComboBox.SelectedIndex = 0;
        }

        /// <summary>
        /// Click on the start button
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void startButton_Click(object sender, EventArgs e)
        {
            // Create the parameters for the worker
            ReadWorkerParams workerParams = new ReadWorkerParams();
            workerParams.moduleIndex = (byte)this.moduleIndexComboBox.SelectedIndex;
            workerParams.channelIndex = (byte)this.channelIndexComboBox.SelectedIndex;
            workerParams.gain = (byte)this.gainComboBox.SelectedIndex;
            workerParams.convertingTime = (uint)(this.convertingTimeComboBox.SelectedIndex + 1)* 20;
            workerParams.board = this.board;

            if (displayPressureCheckBox.Checked == true)
                workerParams.dataFormat = 1;
            else
                workerParams.dataFormat = 0;

            try
            {
                workerParams.sensorOffset = Convert.ToDouble(this.sensorOffsetTextBox.Text);
                workerParams.sensorSensibility = Convert.ToDouble(this.sensorSensibilityTextBox.Text);
                if (workerParams.sensorSensibility == 0.0)
                {
                    MessageBox.Show("Sensibility cannot be 0.0");
                    return;
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString());
                return;
            }

            this.readWorker.RunWorkerAsync((object)workerParams);
            this.startButton.Enabled = false;
            this.stopButton.Enabled = true;
            this.displayPressureCheckBox.Enabled = false;
        }

        /// <summary>
        /// Start to work
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void readWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker worker = sender as BackgroundWorker;
            ReadWorkerParams workerParams = (ReadWorkerParams)e.Argument;

            for (; ; )
            {
                // Must quit ?
                if (worker.CancellationPending)
                {
                    return;
                }
                else
                {
                    double value = 0.0;
                    int returnValue = 0;

                    System.Threading.Thread.Sleep(100);

                    if ( ( returnValue = board.Read1PressureChannel(workerParams.moduleIndex,
                        workerParams.channelIndex,
                        workerParams.gain,
                        workerParams.dataFormat,
                        workerParams.sensorOffset,
                        workerParams.sensorSensibility,
                        workerParams.convertingTime,
                        0,
                        out value)) != 0)
                    {
                        worker.ReportProgress(1, (object)returnValue);
                        return;
                    }
                    else
                    {
                        worker.ReportProgress(2, (object)value);
                    }
                }
            }
        }

        /// <summary>
        /// The worker changed its status
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void readWorker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            if (e.ProgressPercentage == 1)
            {
                MessageBox.Show("Error while reading Pressure : " + (int)e.UserState);
                this.startButton.Enabled = true;
                this.stopButton.Enabled = false;
            }
            else if (e.ProgressPercentage == 2)
            {
                double value = (double) e.UserState;
                this.valueLabel.Text = String.Format("{0:0.000000}",value) + ((this.displayPressureCheckBox.Checked == true) ? "" : " V");
            }
        }

        /// <summary>
        /// Click on stop
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void stopButton_Click(object sender, EventArgs e)
        {
            this.readWorker.CancelAsync();
            this.startButton.Enabled = true;
            this.stopButton.Enabled = false;
            this.displayPressureCheckBox.Enabled = true;
        }

        private void displayPressureCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            if (displayPressureCheckBox.Checked)
            {
                sensorOffsetTextBox.Enabled = true;
                sensorSensibilityTextBox.Enabled = true;
            }
            else
            {
                sensorOffsetTextBox.Enabled = false;
                sensorSensibilityTextBox.Enabled = false;
            }
        }
    }
}
