using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Globalization;

namespace Scan
{
    public partial class Scan : Form
    {
        /// <summary>
        /// Handle to the first board
        /// </summary>
        private PCI3300_NET.PCI3300 board;

        /// <summary>
        /// Board number of modules
        /// </summary>
        private uint _numberOfModules;
        
        /// <summary>
        /// Sensor offset table with offsets in mV
        /// </summary>
        private double[] _sensorOffset;

        /// <summary>
        /// Sensor sensibility array with Sensibility in mV/V/Unit (Unit depends on the sensor)
        /// </summary>
        private double[] _sensorSensibility;

        /// <summary>
        /// Constructor of the form
        /// </summary>
        public Scan()
        {
            InitializeComponent();
            _numberOfModules = 0;
            _sensorOffset = new double[2];
            _sensorSensibility = new double[2];
        }

        /// <summary>
        /// Called at the first load
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Scan_Load(object sender, EventArgs e)
        {
            int returnValue = 0;

            try
            {
                // Open the first board of the system
                this.board = PCI3300_NET.PCI3300.OpenBoard(0);
                // Error handling
                if (this.board == null)
                {
                    MessageBox.Show("Error OpenBoard returned null");
                    this.Close();
                    return;
                }

                // Install the interrupt routine
                returnValue = this.board.SetBoardIntRoutine();
                if (returnValue != 0)
                {
                    MessageBox.Show("SetBoardIntRoutine error : " + returnValue.ToString());
                    this.Close();
                    return;
                }

                // Get the board number of modules
                returnValue = this.board.GetNumberOfModules(out _numberOfModules);
                if (returnValue != 0)
                {
                    MessageBox.Show("GetNumberOfModules error : " + returnValue.ToString());
                    this.Close();
                    return;
                }
            }
            catch(Exception ex)
            {
                MessageBox.Show(ex.ToString());
                return;
            }
            
            // Add modules to combobox
            for (int i = 0; i < _numberOfModules; i++)
                moduleIndexComboBox.Items.Add(i.ToString());

            Thread.CurrentThread.CurrentCulture = new CultureInfo("en-US");

            // Listener to the interrupt
            this.board.ScanEventReceived += new PCI3300_NET.PCI3300.ScanEventHandler(board_ScanEventReceived);
            
            moduleIndexComboBox.SelectedIndex = 0;
            convertingTimeComboBox.SelectedIndex = 0;
            triggerComboBox.SelectedIndex = 0;
            delayUnitComboBox.SelectedIndex = 0;
        }

        /// <summary>
        /// Occurs when a scan result is available
        /// </summary>
        /// <param name="source"></param>
        /// <param name="args"></param>
        void board_ScanEventReceived(IntPtr handle, Byte overflow, UInt32 mask, double[] args, UInt32 argsSize)
        {
            try
            {
                if (displayPressureCheckBox.Checked == true)
                {
                    double[] pressureValue = new double[2];
                    for (int i = 0; i < 2; i++)
                    {
                        if (board.ConvertAnalogToPressure(args[i], _sensorOffset[i], _sensorSensibility[i], out pressureValue[i]) != 0)
                        {
                            MessageBox.Show("Error sensor sensibility cannot be 0.0");
                            stopButton_Click(null, null);
                        }
                    }
                    pressureValue0Label.Text = pressureValue[0].ToString("0.000000");
                    pressureValue1Label.Text = pressureValue[1].ToString("0.000000");
                }
                else
                {
                    pressureValue0Label.Text = args[0].ToString("0.000000") +" V";
                    pressureValue1Label.Text = args[1].ToString("0.000000") +" V";
                }
            }
            catch (Exception) { };
        }

        #region "Start/Stop event"

        /// <summary>
        /// Click on start
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void startButton_Click(object sender, EventArgs e)
        {
            int returnValue = 0;
            UInt16 delayTime = 0;

            try
            {
                delayTime = Convert.ToUInt16(delayValueTextBox.Text);
                if (delayCheckBox.Checked == true)
                {
                    if (delayTime <= (5000 / ((1 << convertingTimeComboBox.SelectedIndex) * 20)))
                    {
                        MessageBox.Show("Delay value must be greater than 5 * 1 / (converting time value)");
                        return;
                    }
                }
                _sensorOffset[0] = Convert.ToDouble(channel0SensorOffsetTextBox.Text);
                _sensorOffset[1] = Convert.ToDouble(channel1SensorOffsetTextBox.Text);
                _sensorSensibility[0] = Convert.ToDouble(channel0SensorSensibilityTextBox.Text);
                _sensorSensibility[1] = Convert.ToDouble(channel1SensorSensibilityTextBox.Text);
            }
            catch (Exception ex)
            {

                MessageBox.Show(ex.ToString());
                return;
            }


            // Then we start the scan acquisition on this module
            returnValue = this.board.InitAndStartScanModuleAcquisition((byte)moduleIndexComboBox.SelectedIndex,
                (uint)((1 << convertingTimeComboBox.SelectedIndex) * 20),
                (byte)((delayCheckBox.Checked) ? 1 : 0),
                delayTime,
                (UInt16)delayUnitComboBox.SelectedIndex,
                (byte)((continuousScanCheckBox.Checked) ? 1 : 0),
                (byte)triggerComboBox.SelectedIndex);
            if (returnValue != 0)
            {
                MessageBox.Show("StartScanModuleAcquisition error : " + returnValue.ToString());
                return;
            }

            stopButton.Enabled = true;
            startButton.Enabled = false;
            delayGroupBox.Enabled = false;
            sensorGroupBox.Enabled = false;
            moduleConfigurationGroupbox.Enabled = false;
        }

        /// <summary>
        /// Click on stop
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void stopButton_Click(object sender, EventArgs e)
        {
            int returnValue = this.board.StopScanModuleAcquisition((byte)moduleIndexComboBox.SelectedIndex);
            if (returnValue != 0)
            {
                MessageBox.Show("StopScanModuleAcquisition error : " + returnValue.ToString());
                return;
            }

            stopButton.Enabled = false;
            startButton.Enabled = true;
            delayGroupBox.Enabled = true;
            sensorGroupBox.Enabled = true;
            moduleConfigurationGroupbox.Enabled = true;
        }
        #endregion

        /// <summary>
        /// Occurs when the form is beeing closed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Scan_FormClosing(object sender, FormClosingEventArgs e)
        {
            // Stop all the scan
            for (byte i = 0; i < 4; i++)
            {
                this.board.StopScanModuleAcquisition(i);
            }

            // Reset the board interrupt routine
            this.board.ResetBoardIntRoutine();
        }

        private void displayPressureCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            if (displayPressureCheckBox.Checked == true)
            {
                channel0SensorOffsetTextBox.Enabled = true;
                channel0SensorSensibilityTextBox.Enabled = true;
                channel1SensorOffsetTextBox.Enabled = true;
                channel1SensorSensibilityTextBox.Enabled = true;
            }
            else
            {
                channel0SensorOffsetTextBox.Enabled = false;
                channel0SensorSensibilityTextBox.Enabled = false;
                channel1SensorOffsetTextBox.Enabled = false;
                channel1SensorSensibilityTextBox.Enabled = false;
            }
        }

        private void delayCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            if (delayCheckBox.Checked == true)
            {
                delayUnitComboBox.Enabled = true;
                delayValueTextBox.Enabled = true;
            }
            else
            {
                delayUnitComboBox.Enabled = false;
                delayValueTextBox.Enabled = false;
            }
        }
    }
}
