using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace Set3DigitalOutputs
{
    public partial class Set3DigitalOutputs : Form
    {
        /// <summary>
        /// Board object
        /// </summary>
        private PCI3300_NET.PCI3300 board;

        /// <summary>
        /// Useful array to access all the checkboxes
        /// </summary>
        private CheckBox[] outputCheckBoxes;

        /// <summary>
        /// Enables to lock the events
        /// </summary>
        private bool locked;

        /// <summary>
        /// Constructor of the form
        /// </summary>
        public Set3DigitalOutputs()
        {
            InitializeComponent();

            // Construct the useful array
            this.outputCheckBoxes = new CheckBox[3];
            this.outputCheckBoxes[0] = this.output0CheckBox;
            this.outputCheckBoxes[1] = this.output1CheckBox;
            this.outputCheckBoxes[2] = this.output2CheckBox;
        }

        /// <summary>
        /// Occurs the first time the form loads
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Set3DigitalOutputs_Load(object sender, EventArgs e)
        {
            int returnValue = 0;

            // Construct the board object
            this.board = PCI3300_NET.PCI3300.OpenBoard(0);

            // Check the result
            if (this.board == null)
            {
                MessageBox.Show("Error while opening board 0");
                this.Close();
                return;
            }

            // Set the output memory to off
            if ((returnValue = board.SetDigitalOutputMemoryOff()) != 0)
            {
                MessageBox.Show("Error while setting output memory to off : " + returnValue);
                this.Close();
                return;
            }

            // Update values of the outputs
            UpdateOutputValues();
        }

        /// <summary>
        /// Update the output values
        /// </summary>
        private void UpdateOutputValues()
        {
            byte outputStatus = 0;
            int returnValue = 0;

            // Get the current status of the outputs
            if ((returnValue = board.Get3DigitalOutputsStatus(out outputStatus))!= 0)
            {
                MessageBox.Show("Error while getting status of the 3 digital outputs : " + returnValue);
                return;
            }

            // Lock
            locked = true;

            // Change the status of the checkboxes
            for (int i = 0; i < 3; i++)
            {
                if ( (uint)(outputStatus & (1 << i)) == (uint)(1 << i))
                {
                    this.outputCheckBoxes[i].Checked = true;
                }
                else
                {
                    this.outputCheckBoxes[i].Checked = false;
                }
            }

            // Unlock
            locked = false;
        }

        /// <summary>
        /// Change the checkbox value
        /// </summary>
        /// <param name="index"></param>
        private void CheckBoxChanged(int index)
        {
            int returnValue = 0;

            // Changment due to update value function ?
            if (locked == true)
            {
                return;
            }

            if (this.outputCheckBoxes[index].Checked == true)
            {
                // Set to on
                if ((returnValue = board.Set3DigitalOutputsOn((byte)(1 << index))) != 0)
                {
                    MessageBox.Show("Error while setting 3 digital outputs to on : " + returnValue);
                }
            }
            else
            {
                // Set to off

                // Check the status of the output memory
                if (outputMemoryCheckBox.Checked == true)
                {
                    if ((returnValue = board.Set3DigitalOutputsOff((byte)(1 << index))) != 0)
                    {
                        MessageBox.Show("Error while setting 3 digital outputs to off : " + returnValue);
                    }
                }
            }

            // Update values
            this.UpdateOutputValues();
        }

        private void output0CheckBox_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckBoxChanged(0);
        }

        private void output1CheckBox_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckBoxChanged(1);
        }

        private void output2CheckBox_CheckedChanged(object sender, EventArgs e)
        {
            this.CheckBoxChanged(2);
        }

        /// <summary>
        /// Change the state of the output memory
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void outputMemoryCheckBox_CheckedChanged(object sender, EventArgs e)
        {
            int returnValue = 0;

            if (outputMemoryCheckBox.Checked == true)
            {
                // Set to on
                if ((returnValue = board.SetDigitalOutputMemoryOn()) != 0)
                {
                    MessageBox.Show("Error while setting digital output memory to on : " + returnValue);
                }
            }
            else
            {
                // Set to off
                if ((returnValue = board.SetDigitalOutputMemoryOff()) != 0)
                {
                    MessageBox.Show("Error while setting digital output memory to off : " + returnValue);
                }
            }
        }
    }
}